/*
   calmonsvr : Local Station Software - Pierre Auger Project
   ***********
                             ----------------
              JM Brunet, L Gugliemi, C Le Gall, G Tristram
	               College de France - Paris

   Created    08/09/99 - JMB
   Modified : 15/06/00 - Version 2.4 : Monitoring part

----------------------------------------------------------------------------*/

#include <stdio.h>
#include <alarm.h>
#include <types.h>
#include <errno.h>

#define MAIN

#include "msgdefs.h"
#include "ready_acq.h"
#include "augererr.h"
#include "feevts.h"

#define _STATUS_MAIN_
#include "status.h"
#define _MONITOR_MAIN_
#define _CONFIG_MAIN_
#include "run_config.h"
#include "central_local.h"

#define MYNAME CALMON

char StrLog[256];
signal_code Signal= 0;

extern int errno; 

PROCESS *Pproc;
MESSAGE *pMessIn, *pMessOut;
unsigned int MonitIdent;


void init( void);
void byebye( error_code err);
void monitoring( MsgTypeOut type, int itag);
void calibration( MsgTypeOut type, int itag);
void AskBufMbx( MailboxId mbx_out, int Bsize_out);
void dmp( unsigned char *p, int size);
error_code SendMessage( MailboxId mbxid, int size,
			MsgTypeOut type, char version);


void sig_hand( signal_code s )
{
  int n;
  Signal= s;
  if( Signal == SIGQUIT) byebye( 0);
  _os_rte();
}

main( int argc, char **argv )
{
  int lsize;
  unsigned int nt=2;
  error_code err;
  signal_code dummy;

  init();
  while( 1) {
    nt= 0; _os_sleep( &nt, &dummy);   /* Wait for a Signal */
    switch ( Signal) {
    case SIG_MONITOR:                 /* Signal from Monitor */
      MonitIdent= 0;
      monitoring( M_MONIT_SEND, TAG_SEND);
      break;
    case SIG_CALIB:                   /* Signal from Calib */
      calibration( M_CALIB_SEND, TAG_SEND);
      break;
    case SIG_CALMON:                  /* Signal from MsgSvrIn */
      if((err= PostReceive( Pproc->mbx_in, (void **)&pMessIn, &lsize))
	    == E_SUCCESS) {
	switch( pMessIn->mheader.type ) {
	case M_MONIT_REQ:
	  MonitIdent= *(unsigned int *)pMessIn->buf;
	  monitoring( M_MONIT_REQ_ACK, TAG_REQ_ACK);
	  break;
	case M_CALIB_REQ:
	  calibration( M_CALIB_REQ_ACK, TAG_REQ_ACK);
	  break;
	default:
	  sprintf( StrLog, "calmonsvr: Message type %d unknown\n", 
			     pMessIn->mheader.type);
	  PrtLog( StrLog, 1);
	}
	PostCancel( Pproc->mbx_in, pMessIn);
	break;
      }
      else if( err != POST_ERR_EMPTY) {
	sprintf( StrLog, "calmonsvr: %s\n", PostError( err));
	PrtLog( StrLog, 1);
      }
      break;
    default: 
      sprintf( StrLog, "calmonsvr: Unknown signal %d received\n", Signal);
      PrtLog( StrLog, 1);
      break;
    }
  }
}



void init( void)
{
  error_code err;
  signal_code sig;


                                   /* Link to Data Modules        */
  if( dm_init( StatModName, sizeof( STATUS), (void **)&StData, &StHeader)) {
    printf( "calmonsvr: **** Can't create %s data module (errno= %d) ****\n", 
      StatModName, errno);
    exit( (int)0);
  }
  StData->calmon.PidCalMon= getpid();
  if( dm_init( ConfigModName, sizeof( CONFIG), 
	       (void **)&CfData, &CfHeader)) {
    sprintf( StrLog, "calmonsvr: Can't create %s DM (err= %d)\n", 
	     ConfigModName, errno);
    PrtLog( StrLog, 1);
    exit( (int)0);
  }
                                         /* Monit Buffer initialisation */
  while(( err= EvbInit(&MonitId, MONIT_BUFFER_NAME, MONIT_SIZE)) == E_KWNMOD);
  if( err) {
    sprintf( StrLog, "calmonsvr: Error $%x EvbInit\n", err);
    PrtLog( StrLog, 1);
    byebye( err);
  }
  if(( Pproc= myprocess( MYNAME)) == NULL) byebye( 1);
                                   /* Open Mailbox in read mode */
  if(( err= ReadyInit( &ReadyEvId, ReadyEvName))) {
    sprintf( StrLog, "calmonsvr: ReadyInit err= %d\n", err);
    PrtLog( StrLog, 1);
    byebye( err);
  }
  
  if((err= PostCreate( &Pproc->mbx_in, Pproc->pnamein, Pproc->Msize_in,
                       Pproc->notify))) {
    sprintf( StrLog, "calmonsvr: Can't Create MBX %s (REC): %d\n", 
	     Pproc->pnamein, err ) ;
    PrtLog( StrLog, 1);
    byebye( err);
  }

  if(( err= ReadyToGo( ReadyEvId, READY_MONITOR))) {
    sprintf( StrLog, "calmonsvr : ReadyToGo(): err= %d\n", err);
    PrtLog( StrLog, 1);
    byebye( err);
  }

  if(( err= ReadyAll( ReadyEvId, &sig, READY_MSGSVROUT)) != E_SUCCESS) {
    sprintf( StrLog, "calmonsvr: Unexpect sig %d in ReadyAll(): err= %d\n",
	     sig, err);
    PrtLog( StrLog, 1);
    byebye( err);
  }

                                   /* Open Mailbox in send mode */
  if ( (err = PostLink( &Pproc->mbx_out, Pproc->pnameout ))) {
    sprintf( StrLog, "calmonsvr : Can't Open MBX %s\ (Send): %d\n",
            Pproc->pnameout , err ) ;
    PrtLog( StrLog, 1);
    byebye( err);
  }

                                   /* Intercept Signals */
  _os_intercept( sig_hand, _glob_data );
  err= ReadyFinish( ReadyEvId, ReadyEvName);
  printf( "calmonsvr : init ended...\n");
  fflush( stdout);
}


void byebye( error_code err)
{
  dm_finish( &StHeader );
  dm_finish( &CfHeader );
  exit( 0);
}


void AskBufMbx( MailboxId mbx_out, int Bsize_out)
{
  error_code err;
  signal_code dummy;
  u_int32 t;

  while(( err= PostAlloc( mbx_out, (void **)&pMessOut, Bsize_out)) 
	!= E_SUCCESS) {
    if( err == POST_ERR_FULL) {
      t= 1; _os_sleep( &t, &dummy);         /* Mailbox full, try again */
      continue;
    }                                       /* Fatal error !!!!        */
    sprintf( StrLog, "calmonsvr: Can't allocate a MBX buffer\n");
    PrtLog( StrLog, 1);
    sprintf( StrLog, "%s\n", PostError( err));
    PrtLog( StrLog, 1);
    byebye( err);
  }
}

void monitoring( MsgTypeOut type, int itag)
{
  int size= 8, err, n;
  char version= 1;
  u_int32 *mesbufout;
  MONITOR Monit, *pmonit= &Monit;
                    /* Ask for a buffer in Mailbox-out */
  AskBufMbx( Pproc->mbx_out, Pproc->Bsize_out);
  mesbufout= (u_int32 *)pMessOut->buf;
  memcpy( (void *)mesbufout, (void *)&MonitIdent, 4);
  mesbufout++;
                    /* Ask for a Monitoring buffer */
  err = EvbGetLastTag(MonitId, pmonit, &n, itag, 0, 1);
  if (err) {
    *mesbufout= NO_MONIT_BUFFER;
    printf( "calmonsvr: ***** Monitoring buffer not available *****\n");
  } 
  else {
    *mesbufout++= BUFFER_OK;
    memcpy( (void *)mesbufout, (void *)pmonit, n);
    size += n;
  }
                    /* Put data into buffer */
                    /* send message to MsgSvrOut */
  SendMessage( Pproc->mbx_out, size, type, version);
}


void calibration( MsgTypeOut type, int itag)
{
}

void dmp( unsigned char *p, int size)
{
  int n;

  printf( "calmon dump : size= %d", size); fflush( stdout);
  for( n=0; n<size; n++) {
    if( !(n % 20)) printf( "\n");
    printf( "%3X", *p++);
  }
  printf( "\n");
  fflush( stdout);
}    


error_code SendMessage( MailboxId mbxid, int size,
			MsgTypeOut type, char version)
{
  error_code err;

  pMessOut->mheader.length= size+ sizeof(MESSAGE_HEADER);
  pMessOut->mheader.type=   type;
  pMessOut->mheader.version= version;
  if(( err= PostSendShrink( mbxid, pMessOut, size+sizeof(MESSAGE_HEADER))) 
     != E_SUCCESS) {
    sprintf( StrLog, "calmonsvr : %s\n", PostError( err));
    PrtLog( StrLog, 1);
  }
}
