/* 
   control : Local Station Software - Pierre Auger Project
   ---------
              JM Brunet, L Gugliemi, G Tristram
	               College de France - Paris
	      Patrick Allison
	               Penn State University

   Created    01/06/97 - JMB & CL
   Modified : 17/10/97 - Split MsgSvr into MsgSvrIn & MsgSvrOut
              16/11/98 - Version 2
	      03/03/98 - Version 2.2
	      22/02/00 - Version 2.3 : Add communication with Suscriber Unit
                                       Change in Frame & Header formats
	      25/05/00 - Version 2.4 : Fake Flash2Config()
	      Aug 2000 - Version 2.5 : Logfile send
	      Sep 2000 - Version 2.6 : Download file & OS9 command
	      Oct 2000 -               RS-232 Comm. Protocol V2.0
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <process.h>
#include <errno.h>
#include <cglob.h>
#include <time.h>

#define MAIN
#define _CONFIG_MAIN_
#define _MONITOR_MAIN_
#define MYNAME  CONTROL
#define _STATUS_MAIN_


#include "msgdefs.h"
#include "dmlib.h"
#include "ready_acq.h"
#include "status.h"
#include "central_local.h"
#include "central.h"
#include "augererr.h"
#include "evbdefs.h"
#include "run_config.h"
#include "svrdm.h"

/*============== Definition of tasks to fork : ========*/
typedef struct {
  char name[80];       	/* Process name */
  process_id pid;	   	/* Process ID */
  u_int16 prior;	   	/* Process priority */
} PROC;

PROC Proc[]= {
  "msgsvrin",  0, 128,
  "msgsvrout", 0, 130,
  "trigger2",  0, 130,
  "mufill",    0, 130,
  "evtsvr",    0, 128,
  "calib",     0, 120,
  "monitor",   0, 120,
  "calmonsvr", 0, 120,
  "gpsctrl",   0, 120,
  "tcpsvr",    0, 115
};

int ProcNb= 10;			/* Total number of processes to fork */

/*============== End definition of tasks to fork : ========*/

extern errno;

/*============== Logging errors : ========*/

FILE *Prevlog= NULL;
char PrevLogFileName[]= "prev_error.log";
char StrLog[256];

/*============== Subroutine Definitions =====================
void init(void):
	Initialization subroutine. Run once.

void create_event( event_id *EventId, char *EventName):
	Create a new OS9 Event with name located at EventName.
	Store its event ID at EventId.

void unlink_delete( event_id EventId, char *EventName):
	Reduce the link count of the event named at EventName
	with ID EventId and delete it if the link count is 0.

void wait_no_event( event_id evt_id, char *evt_name):
	Wait for the event named at evt_name with ID evt_id
	to have a value of 0. 

void fork_task(void):
	Fork all of the tasks listed in Proc[].

error_code ConfigSetting( unsigned char type):
	Change configuration settings: do either
		M_CONFIG_SET: update configuration with sent parameters
		M_FLASH_TO_CONFIG: restore configuration from EPROM
		M_CONFIG_TO_FLASH: save configuration in EPROM (not impl.)
	Returns error_code describing success/failure (not impl.)

void byebye( error_code err):
	Exits with error code err.

int ResetStatus( int flag):
	Re-initializes the status module: if flag is 0, run number
	is re-initialized as well, otherwise the run number is incremented.
	For now, returns 1.

int Flash2Config( void):
	Transfers the values of the Flash EPROM into the Config module.
	For now, returns 1. (not impl.)

error_code SendMessage(MailboxId mbxid,int size,MsgTypeOut type, char version):
	Sends a message of type type through the mbxid mailbox.

void AskBufMbx( MailboxId mbx_out,int Bsize_out);
	Request	an additional buffer in the mbx_out mailbox (global variable
	*pMessOut) of size Bsize_out. 
	Wait if not available, exit on other error.

void StartT1( void):
	Start trigger 1. (not impl.)

void StopT1(void):
	Stop trigger 1. (not impl.)

void SendSig( MSGSIG sig, char *name):
	Send the signal sig to the process whose name is at name.

int SendPrevLogFile( char *):
        Send the previous logfile to CS (message M_LOG_SEND)

int Download( void):
        File is Downloaded from CS and stored on RAM Disk
int OS9Cmd( void):
        A command is received from CS and executed. Then, the
	command return code is sent to CS.
int ReadyMessage( unsigned char *p):
        Prepare data part of the M_READY message.
	Return message nb of bytes 
/=============================================================*/


void init( void);
void create_event(  event_id *EventId, char *EventName);
void unlink_delete( event_id  EventId, char *EventName);
void wait_no_event( event_id evt_id, char *evt_name);
void fork_task( void);
error_code ConfigSetting( unsigned char type);
void byebye( error_code);
int  ResetStatus( int flag);
int  Flash2Config( void);
int  ResetSvr( void);
error_code SendMessage( MailboxId mbxid, int size, 
			MsgTypeOut type, char version);
void AskBufMbx( MailboxId mbx_out, int Bsize_out);
void StartT1( void);
void StopT1( void);
void SendSig( MSGSIG sig, char *name);
int  SendPrevLogFile( char *);
int  Download( void);
int  OS9Cmd( void);
void dmp( char * title, unsigned char *p, int size);
int ReadyMessage( unsigned char *p);

STATUS *st; /* Global status module */
TRIG1  *t1;	/* Trigger 1 Module */
CONFIG *cf;	/* Global config module */
PROCESS *Pproc;	/* Process identifier for this process */

signal_code Signal = 0 ;  /* Variable for received signals */

process_id PID_ro, PID_ca, PID_t2, PID_sv; /* Process IDs for */
char *argblk[]= { 0,0,0,0,0,0,};

int TotalSig=0, Nmess= 10, Debug= 11;
MESSAGE *pMessIn, *pMessOut; /* Mailbox pointers */



/* =================== Signal handler =================*
 * Store received signal in global variable Signal;    *
 * If the signal received is a mailbox Message Arrived *
 * signal (SIG_CTRL), increment the TotalSig global var*
 * ====================================================*/


void
sig_hand( signal_code s )
{
  Signal= s;
  if( s == Pproc->notify->param1) TotalSig++;
  _os_rte();
}



main( int argc, char *argv)
{
  int lsize, flag= 0, n;
  signal_code dummy;
  u_int32 t;
  error_code err;
  u_int32 *mesbufout;
  unsigned char *p1, *p2;
  char bell= 7;

/* Initialize buffers, modules, fork other tasks */
  init();

/* Start infinite Loop         */
  while( 1) {                           

/* Debugging information: 
	exit on SIGQUIT, print last signal, print # of messages waiting */
    if( Signal == SIGQUIT) byebye( 2000);

/* Process ALL incoming messages */
    while(TotalSig)
      {
	 /* Process ONE incoming message */
	if((err= PostReceive( Pproc->mbx_in, (void **)&pMessIn, &lsize))
	   == E_SUCCESS) {
	  TotalSig--;
                              	/* Ask for a buffer in Mailbox-out */
	  if( pMessIn->mheader.completion == COMPLETION_ALL ||
	      pMessIn->mheader.completion == COMPLETION_FIRST) {
	    AskBufMbx( Pproc->mbx_out, Pproc->Bsize_out);
	    mesbufout= (u_int32 *)pMessOut->buf;
	  }

/* Determine message type and content */
	  switch( pMessIn->mheader.type ) {

/*=============================================================*
 * M_WAKEUP                  :CS->LS: Wake up local station    *
 *                                                             *
 * Action:                                                     *
 *        Start Trigger 1.                                     *
 *        Set IsFast1Enabled bit in cf->FeParams               *
 *        Send M_READY message to CS                           *
 *                                                             *
 *=============================================================*/

 	  case M_WAKEUP:
	    StartT1();                       /* 28/10/99 */
	    cf->FeParams.IsFast1Enabled= 1;  /* 28/10/99 */
	    n= ReadyMessage( pMessOut->buf); /* 16/10/00 */
	    SendMessage( Pproc->mbx_out, 0, M_READY, n);
	    break;

/*=============================================================*
 * M_RUN_START_REQ           :CS->LS: Begin run                *
 *                                                             *
 * Action:                                                     *
 *        Check RunEnable bit under cf->AcqParams              *
 *        Set RunStatus in st module to START_RUN_REQUESTED    *
 *        Store the run start time in fmr.CpuStartRun          *
 *        Initialize the Status data module                    *
 *        Set the IsFast0Enabled bit under cf->FeParams        *
 *        Set the IsFast1Enabled bit under cf->FeParams        *
 *        Execute the M_RUN_CONTINUE_REQ case as well          *
 *                                                             * 
 *=============================================================*/

	  case M_RUN_START_REQ:
	    if( !cf->AcqParams.RunEnable) {  /* 21/03/2000 */
	      printf( "control: RunEnable unset : Start run cmd ignored\n");
	      sprintf( StrLog, 
		      "control: RunEnable unset : Start run cmd ignored\n");
	      PrtLog( StrLog, 0);
	      break;
	    }
	    st->RunStatus= START_RUN_REQUESTED;
	    st->fmr.CpuStartRun= time( NULL);
	    ResetStatus( flag++);    /* Status data module initialization */
	    cf->FeParams.IsFast0Enabled= 1;
	    cf->FeParams.IsFast1Enabled= 1;

/*=============================================================*
 * M_RUN_CONTINUE_REQ        :CS->LS: Continue a paused run    *
 *                                                             *
 * Action:                                                     *
 *        Set RunStatus to RUN_STARTED                         *
 *        Start Trigger 1                                      *
 *        Send either M_RUN_START_ACK (if M_RUN_START_REQ)     *
 *             or     M_RUN_CONTINUE_ACK                       *
 *                                                             *
 *=============================================================*/


	  case M_RUN_CONTINUE_REQ:
	    st->RunStatus= RUN_STARTED;
	    StartT1();
	    if( pMessIn->mheader.type == M_RUN_START_REQ) {
	      SendMessage( Pproc->mbx_out, 0, M_RUN_START_ACK, 0);
	      t1->RunNumber++;
	    }
	    else SendMessage( Pproc->mbx_out, 0, M_RUN_CONTINUE_ACK, 0);
	    break;

/*=============================================================*
 * M_RUN_STOP_REQ            :CS->LS: Stop an active run       *
 *                                                             *
 * Action:                                                     *
 *        Set RunStatus to RUN_STOPPED                         *
 *        Send SIG_STOP to trigger 2                           *
 *        Send M_RUN_STOP_ACK to CS                            *
 *                                                             *
 *=============================================================*/

	  case M_RUN_STOP_REQ:
	    st->RunStatus= RUN_STOPPED;
	    SendSig( SIG_STOP, "trigger2");
	    SendMessage( Pproc->mbx_out, 0, M_RUN_STOP_ACK, 0);
	    break;

/*=============================================================*
 * M_RUN_PAUSE_REQ           :CS->LS: Pause an active run      *
 *                                                             *
 * Action:                                                     *
 *        Set RunStatus to PAUSE_RUN_REQUESTED                 *
 *        Stop Trigger 1                                       *
 *        Sleep for ~ 5 clock ticks to wait for Trig1 stop     *
 *        Wait for all muon triggers to complete               *
 *        Wait for all Trigger 2s to complete                  *
 *        Set RunStatus to RUN_PAUSED                          *
 *        Send M_RUN_PAUSE_ACK to CS                           *
 *                                                             *
 *=============================================================*/

	  case M_RUN_PAUSE_REQ:
	    st->RunStatus= PAUSE_RUN_REQUESTED;
	    StopT1();
	    t=5; _os_sleep( &t, &dummy );
	                           /* Wait for all triggers to be processed */
	    wait_no_event( st->event.IdT1Mu, st->event.EvNameT1Mu);
	    wait_no_event( st->event.IdT1Trigger2, st->event.EvNameT1Tg2);
	    SendSig( SIG_PAUSE, "trigger2");
	    st->RunStatus= RUN_PAUSED;
	    SendMessage( Pproc->mbx_out, 0, M_RUN_PAUSE_ACK, 0);
	    break;

/*=============================================================*
 * M_LOG_REQ           :CS->LS: Request previous error logfile *
 *                                                             *
 * Action:                                                     *
 *        open logfile                                         *
 *        put it in PrevLog table                              *
 *        send M_LOG_SEND message                              *
 *=============================================================*/

	  case M_LOG_REQ:
	    SendPrevLogFile( (char *)pMessOut->buf);
	    break;
/*=============================================================*
 * M_CONFIG_SET              :CS->LS: Set config parameters    *
 * M_FLASH_TO_CONFIG         :CS->LS: Restore config parameters*
 * M_CONFIG_TO_FLASH         :CS->LS: Save config parameters   *
 *                                                             *
 * Action:                                                     *
 *        Pass message header to ConfigSetting to set config   *
 *        Send M_CONFIG_SET_ACK with any error message to CS   *
 *                                                             *
 *=============================================================*/

	  case M_CONFIG_SET: case M_FLASH_TO_CONFIG: case M_CONFIG_TO_FLASH:
	    err= ConfigSetting( pMessIn->mheader.type);
	    *mesbufout= err;
	    SendMessage( Pproc->mbx_out, sizeof( u_int32), M_CONFIG_SET_ACK,0);
	    break;

/*=============================================================*
 * M_MSG_ERROR               :MsgSvrIn: Error in message frame *
 *                                                             *
 * Action:                                                     *
 *        Copy contents of message into the MessageOut buffer  *
 *        Send M_MSG_ERROR to CS containing error message      *
 *                                                             * 
 *=============================================================*/


	  case M_MSG_ERROR :      /* from MsgSvrIn */
	    pMessOut->mheader= pMessIn->mheader;
	    p1= pMessOut->buf; p2= pMessIn->buf;
	    for( n=0; n< pMessIn->mheader.length; n++) *p1++= *p2++;
	    SendMessage( Pproc->mbx_out, pMessIn->mheader.length, M_MSG_ERROR, 
			 pMessIn->mheader.version);
	    break;

/*=============================================================*
 * M_REBOOT                  :CS->LS: Reboot local station     *
 *                                                             *
 * Action:                                                     *
 *        Set RunStatus to REBOOT                              *
 *        Exit with error 0                                    *
 *                                                             *
 *=============================================================*/

	  case M_REBOOT:
	    PrtLog( "control : M_REBOOT received\n", 0);
	    st->RunStatus= REBOOT;
	    byebye( 0);

/*=============================================================*
 * M_DOWNLOAD                  :CS->LS: Download a file        *
 *                                                             *
 * Action:                                                     *
 *        Receive a file from CS and put it on RamDisk         *
 *                                                             *
 *=============================================================*/

	  case M_DOWNLOAD:
	    Download();
	    break;
	
/*=============================================================*
 * M_OS9_CMD                   :CS->LS: Send an OS9 command    *
 *                                                             *
 * Action:                                                     *
 *        Receive a command, execute it and send return code   *
 *                                                             *
 *=============================================================*/

	  case M_OS9_CMD:
	    OS9Cmd();
	    break;

/*=============================================================*
 * Default case:                                               *
 *                                                             *
 * Action:                                                     *
 *        Send M_UNKNOWN message to CS containing header type  *
 *                                                             *
 *=============================================================*/

	  default:
	    sprintf( StrLog, "control Message type %d unknown\n",
		       pMessIn->mheader.type);
	    PrtLog( StrLog, 1);
	    *mesbufout= pMessIn->mheader.type;
	    SendMessage( Pproc->mbx_out, sizeof( u_int32), M_UNKNOWN, 0);
	    break;
	  } /*** End switch( type) ***/
/* Delete processed message */
	  PostCancel( Pproc->mbx_in, pMessIn);
	} /*** End PostReceived == SUCCESS ***/
	else if( err != POST_ERR_EMPTY) {
	  sprintf( StrLog, "control: %s\n", PostError( err));
	  PrtLog( StrLog, 1);
	}
      } /*** End while( TotalSig) ***/
    t=0; _os_sleep( &t, &dummy );
  } /************ Stop infinite Loop *************************************/

  byebye( 0);
}

/* =================== ResetStatus ====================*
 * Initialize status module and buffers                *
 * If flag is 0, reset the run count, otherwise        *
 * increment the run count                             *
 * ====================================================*/

int ResetStatus( int flag) /* Status data module & Buffers initialization */
{
  st->inter.T1Fast0= 0;
  st->inter.T1Fast1= 0;
  st->inter.T1Random= 0;
  st->inter.T1Slow= 0;
  st->inter.T1DeadTime= 0;
  st->trg2.T1Fast0= 0;
  st->trg2.T1Fast1= 0;
  st->trg2.T2= 0;
  st->trg2.NEvtBusy= 0;
  st->mufill.T1Slow= 0;
  st->calmon.T1Random= 0;
  st->evtsvr.T3Asked =0;
  st->evtsvr.T3Found =0;
  st->evtsvr.T3Already =0;
  st->evtsvr.T3Muon =0;
  st->evtsvr.T3Lost =0;
  st->evtsvr.T3NotFound =0;
  st->evtsvr.T3TooYoung =0;
  st->buffer.FastUsed= 0;
  st->buffer.FastMin= 999999;
  st->buffer.FastMax= 0;
  st->buffer.FastLost= 0;
  st->buffer.SlowUsed= 0;
  st->buffer.SlowMin= 999999;
  st->buffer.SlowMax= 0;
  st->buffer.SlowLost= 0;
  st->buffer.EvbUsed= 0;
  st->buffer.EvbMin= 9999999;
  st->buffer.EvbMax= 0;
  st->buffer.EvbLost= 0;
  st->buffer.MubUsed= 0;
  st->buffer.MubMin= 9999999;
  st->buffer.MubMax= 0;
  st->buffer.MubLost= 0;
  st->fmr.ticks= 2;
  st->fmr.FastBufState= 0;
  st->fmr.SlowBufState= 0;
  st->Freeze=0;
  if( !flag) {
    st->fmr.RunNumber=1;
    st->fmr.bunch=1;
    return 1;
  }
  st->fmr.RunNumber++;
  return 1;
}

/* =================== Flash2Config ===================*
 * Restore Config data module information from Flash   *
 * EPROM. Currently not implemented.                   *
 * 						       *
 * ====================================================*/

int Flash2Config( void) /* Config data module initialization */
                       /* WARNING : must be read from FLASH-EPROM */
{
  int n;
  unsigned short *p;
  GPS_PARAMS  Gps_Params_x = {
    0x00 , 0x14  ,
    0x00 , 0x00  , 0x00 , 0x00 ,
    0x00 ,
    0x0a , 0x7b  , 0x46 , 0x77 ,
    0x00 , 0x81  , 0x77 , 0x70 ,
    0x00 , 0x00  , 0x45 , 0x73 ,
    0x00 
  } ; 

  cf->FeParams.IsFast0Enabled=        0;
  cf->FeParams.IsFast1Enabled=        0;
  cf->FeParams.IsSlowEnabled=         1;   /* Start calibration */
  cf->FeParams.IsRandomEnabled=       0;
  cf->FeParams.fast0.Width=          63;   /* 63 TimeSlices  */
  cf->FeParams.fast0.Occupancy=       5;   /*  1 to 63       */
  cf->FeParams.fast0.Thresh=        100;   /*  1 to 255 (Sum 3 PMTs) */
  cf->FeParams.fast0.Delay=         600;   /*  1 to 1023 */
  cf->FeParams.fast1.Width=          63;   /* 63*4 TimeSlices  */
  cf->FeParams.fast1.Occupancy=       5;
  cf->FeParams.fast1.Thresh=        100;
  cf->FeParams.fast1.Delay=         600;
  cf->FeParams.FastZeroSupThresh=   100;   /*  1 to 255 */
  cf->FeParams.slow.Width=           63;   /*  (100 ms) : 23 bits */
  cf->FeParams.slow.ZeroSupThresh=  100;   /*  1 to 255 */
  cf->FeParams.slow.MuZeroSup=       50;   /*  1 to 255 */
  cf->FeParams.slow.MuGateWidth=     50;   /*  1 to 1023 */
  cf->FeParams.Scaler1LowThresh=    100;   /*  1 to 255 */
  cf->FeParams.Scaler2LowThresh=    100;   /*  1 to 255 */
  cf->FeParams.Scaler3LowThresh=    100;   /*  1 to 255 */
  cf->AcqParams.HardAddress=      0xAA5;
  cf->AcqParams.LogicalAddress=       0;   /* not defined */
  cf->AcqParams.VEMmin=             120;
  cf->AcqParams.Trigger2Algorithm=    0;   /* Fake trigger 2 : random */
  cf->AcqParams.RunEnable=            1;   /* Run is enabled */
  cf->MonitParams.ReadRate=           2;   /* Read monit Rate */
  cf->MonitParams.Times=              5;   /* For monitor average */
  cf->MonitParams.IsSendRegular=      1;   /***** Regular send to CS */
  cf->MonitParams.IsTPC=              0;   /* No TPC board */
  cf->MonitParams.DACinUSE= 0xFF;
  p= cf->MonitParams.Dac;
  for( n=0; n<NDAC; n++) *p++= 0;
  p= cf->MonitParams.MinOrange;
  for( n=0; n<NADC; n++) *p++= 1000;
  p= cf->MonitParams.MaxOrange;
  for( n=0; n<NADC; n++) *p++= 2000;
  p= cf->MonitParams.MinRed;
  for( n=0; n<NADC; n++) *p++= 500;
  p= cf->MonitParams.MaxRed;
  for( n=0; n<NADC; n++) *p++= 3000;
  cf->MonitParams.OutputRegister= 0xBEBE;
  cf->MonitParams.ADCinUSE= 0xF0FFFFFF;
  memcpy(cf->GpsParams.AlarmLim , Gps_Params_x.AlarmLim , 
	 sizeof(Gps_Params_x )  ) ;
  return 1;
}


int ResetSvr( void) /* Svr data module initialization */
{
  SvrData->BSid=0;
  SvrData->LSid= cf->AcqParams.HardAddress;
  SvrData->PacketNumberRec= 0;
  SvrData->PacketNumberSent= 0;
  SvrData->NetworkStatus= 0x02;
  SvrData->IsControlCommand= 0;
  SvrData->IsDataAck=   SvrData->IsSent= 0;
  SvrData->IsSoftReset= SvrData->IsVersionReq= 0;
  SvrData->IsGpsStatusReq= 0;
  SvrData->IsStationIdReq= 0;
  SvrData->IsEchoReq= 0;
  strcpy( SvrData->EchoData, "NULL");
  SvrData->IsCommandInvalid= SvrData->SendCommandInvalid= 0;
  SvrData->IsFail= SvrData->Pid= 0;
}



/* =================== INIT ============================*
 * Initialize Pproc to this process                     *
 * Create Status, Config, Svrdm data modules            *
 * Initialize data modules                              *
 * Create muon trigger event, trigger 2 event           *
 * Create Ready event for synchronization               *
 * Create Control mailbox                               *
 * Initialize buffers					*
 * Fork tasks						*
 * Intercept signals (initialize signal handler)        *
 * Send M_READY to Central Station                      *
 * Launch calibration (not impl.)                       *
 * =====================================================*/

void init( void)
{
  int ret, n;
  int wait_task= 0;
  error_code err;
  time_t tt;
  u_int32 perm= 0x777, tcks;
  signal_code sig;
  unsigned int readymask;
  int status;
  unsigned int t;
  signal_code dummy;
  char cmd[80];


                                    /* For logging purpose */
//  sprintf( cmd, "copy -r error.log %s", PrevLogFileName);
//  system( cmd); sleep( 1);
  system( "date > error.log");
  if(( Pproc= myprocess( MYNAME)) == NULL) {
    printf( "control : error in myprocess()\n");
    PrtLog( "control: error in myprocess()\n", 1);
    exit( (int)0);
  }
                                     /* Create Data Modules        */
  if( dm_init( StatModName, sizeof( STATUS), (void **)&StData, &StHeader))
    exit(_errmsg(errno, "control : Can't create STATUS data module!\n"));
    
  if( dm_init( ConfigModName, sizeof( CONFIG), (void **)&CfData, &CfHeader))
    exit(_errmsg(errno, "control : Can't create CONFIG data module!\n"));
   
  if( dm_init( SvrModName, sizeof( SVRDM), (void **)&SvrData, &SvrHeader))
    exit(_errmsg(errno, "control : Can't create SVRDM data module!\n"));
                  
  st= StData;
  cf= CfData;

                               /* Store permanent info in Status: boot time */
  _os_getime( &tt, &tcks);
  strcpy( st->fmr.StartAcq, ctime( &tt));
  st->fmr.StartAcq[16]= '\0';

  ResetStatus( 0);                   /* Initialize Status data module */
  ConfigSetting( M_FLASH_TO_CONFIG); /* Initialize Config data module */
  ResetSvr();                        /* Initialize Svr    data module */

                                     /* Create trigger events */
  strcpy( st->event.EvNameT1Mu,  "T1Mu");
  strcpy( st->event.EvNameT1Tg2, "T1Tg2");
  create_event( &st->event.IdT1Mu,  st->event.EvNameT1Mu);
  create_event( &st->event.IdT1Trigger2, st->event.EvNameT1Tg2);

                                     /* Let open Mailbox in read mode */
  if(( err= ReadyInit( &ReadyEvId, ReadyEvName))) {
    sprintf( StrLog, "control: Error %d calling ReadyInit\n", err);
    PrtLog( StrLog, 1);

    byebye( err);
  }

                                     /* Create Mailbox, receive mode */
  if((err= PostCreate( &Pproc->mbx_in, Pproc->pnamein, Pproc->Msize_in,
		       Pproc->notify))) {
    sprintf( StrLog, "control: Can't Create Mailbox '%s' for Receive: %d\n", 
	    Pproc->pnamein, err );
    PrtLog( StrLog, 1);
    byebye( err);
  }
  else if( Debug) {
    printf( "control : Mailbox %s ($%X - %d - %d) CREATED\n", 
	    Pproc->pnamein, Pproc->mbx_in, Pproc->Msize_in, 
	    Pproc->notify);
    fflush( stdout);
  }

#ifdef CARD
                                   /* Launch augerirq */
  system( "augerirq&");
  printf( "control : augerirq launched...\n"); fflush( stdout);
#endif

                                   /* Init Fast Buffers */
  if(( err= FB_init( MAX_FAST_BUFFER))) {
    sprintf( StrLog, "control: Can't initialize fast Buffers : error= %d\n",
	       err);
    PrtLog( StrLog, 1);
    byebye( err);
  }
                                   /* Init Slow Buffers */
  if(( err= SB_init( MAX_SLOW_BUFFER))) {
    sprintf( StrLog, "control: Can't initialize slow Buffers : error= %d\n",
	       err);
    PrtLog( StrLog, 1);
    byebye( err);
  }
                                   /* Fork all other tasks  */
  fork_task( );

                                   /* Wait for all maibox created */
  if(( err= ReadyAll( ReadyEvId, &sig, READY_MSGSVROUT)) != E_SUCCESS) {
    sprintf( StrLog, "control: Unexpected signal %d", sig);
    PrtLog( StrLog, 1);
    sprintf( StrLog, "control: when calling ReadyAll() : err= %d\n", err);
    PrtLog( StrLog, 1);
    byebye( err);
  }
  fflush( stdout);

                                   /* Open Mailbox in send mode */
  if ( (err = PostLink( &Pproc->mbx_out, Pproc->pnameout ))) {
    sprintf( StrLog, "control: Can't Open Mailbox \"%s\" for Send : %d\n",
	    Pproc->pnameout , err ) ;
    PrtLog( StrLog, 1);
    byebye( err);
  }
  if(( err= ReadyToGo( ReadyEvId, READY_CONTROL))) {
    sprintf( StrLog, "control: Error when calling ReadyToGo() : %d\n", err);
    PrtLog( StrLog, 1);
    byebye( err);
  }

  t=2; _os_sleep( &t, &dummy);

                                   /* Intercept Signals */
  _os_intercept( sig_hand, _glob_data );
                                   /* Ask for a buffer in Mailbox-out */
  AskBufMbx( Pproc->mbx_out, Pproc->Bsize_out);
                                   /* Send message M_READY to CS */
  st->RunStatus= WAIT_STATE;
  n= ReadyMessage( pMessOut->buf);      /* 16/10/00 */
  SendMessage( Pproc->mbx_out, 0, M_READY, n);

  err= ReadyFinish( ReadyEvId, ReadyEvName);
  printf( "control : M_READY sent to Central Station\n"); fflush( stdout);

  /* Calibration must be started here (SLOW Trigger) */
}

/* =================== ByeBye =========================*
 * Unlink from data modules                            *
 * Send SIGQUIT to all child processes                 *
 * Unlink from trigger events                          *
 * Close Control mailbox                               *
 * Exit with error errno                               *
 * ====================================================*/


void byebye( error_code err)
{
  int32 value;
  u_int32 act_flag= 0;
  status_code status;
  process_id child_id;
  int n, m;
  char child_name[20];
  
  sprintf( StrLog, "control: Ended with error= %d\n", err);
  PrtLog( StrLog, 1);
  err= dm_finish( &StHeader );               /* Unlink Data Module       */
  if( err) printf( "control : dm_finish : error=%d\n", err);
                                             /* Send  Signal to end child */
  for( n=0; n<ProcNb; n++) {
    if(( err= _os_send( Proc[n].pid, SIGQUIT)))
      printf( "control: ERROR _os_send(): %d( proc=%d)\n", err, n);
  }
                                             /* Unlink & Delete Events   */
  unlink_delete( st->event.IdT1Mu,  st->event.EvNameT1Mu);
  unlink_delete( st->event.IdT1Trigger2, st->event.EvNameT1Tg2);
                                             /* Close Mailbox */
  if(( err= PostClose( &Pproc->mbx_out)) != E_SUCCESS)
    printf( "control : Can't close Mailbox %s\n", Pproc->pnameout);
  printf( "control : All is clean!!! Last signal received = %d\n", Signal);
  exit( (int) errno);
}

/* =================== Create_Event ===================*
 * Create event with global RW permissions             *
 * Exit on error                                       *
 *                                                     *
 * ====================================================*/


void create_event( event_id *EventId, char *EventName)
{
  int32 wait_inc= -1, sig_inc= 1, value= 0;
  u_int32 perm= 0x777, color= 0;
  error_code err;
  
  err= _os_ev_creat( wait_inc, sig_inc, perm, EventId, 
		     EventName, value, color);
  if( err) {
    sprintf( StrLog, "control: Can't create event %s : error %s\n", 
	       EventName, strerror( err));
    PrtLog( StrLog, 1);
    byebye( err);
  }
  if( Debug >10) {
    printf( "control : Event %s EvId= %d created\n", EventName, *EventId);
    fflush( stdout);
  }
}

/* =================== Unlink_Delete ==================*
 * Unlink from event EventId- wait 5 times on error    *
 * Delete event (if linkcount is 0)                    *
 *                                                     *
 * ====================================================*/


void unlink_delete( event_id EventId, char *EventName)
{
  error_code err;
  unsigned int t;
  int nb=0;
  signal_code dummy;
                                             /* Unlink Events */
  while( (err= _os_ev_unlink( EventId))) {
    printf( "control : Can't unlink event %s (error %d) ... waitting...\n",
	    EventName, err); fflush( stdout);
    if( ++nb >= 5) break;
    t=2; _os_sleep( &t, &dummy);
  }
                                             /* Delete Events */
  if( (err=_os_ev_delete( EventName))) {
    printf( "control : Can't delete event %s (error= %d)\n",
	    EventName, err); fflush( stdout);
  }
  if( Debug >10) {
    printf( "control : Event %s EvId= %d Deleted\n", EventName, EventId);
    fflush( stdout);
  }
}



void wait_no_event( event_id evt_id, char *evt_name)
{                                    /* Wait for an event to have a value=0 */
  int32 value;
  unsigned int t;
  signal_code dummy;
  int n=10;

  while( 1) {
    n--;
    if( !n) break;
    _os_ev_read( evt_id, &value);
    if( !value) break;
    t=2; _os_sleep( &t, &dummy);
  }
  if( !n) {
    sprintf( StrLog, "control: wait_no_event failed for Event %s (%d)\n", 
		  evt_name, evt_id);
    PrtLog( StrLog, 1);
  }
  else if( Debug >10)
    printf( "control :wait_no_event OK for Event %s (%d)\n", evt_name,evt_id);
  fflush( stdout);
}

/* =================== Fork_Task ======================*
 * For every entry in the process array:               *
 *   Fork process with priority,                       *
 *		pass STDIN,STDOUT,STDERR,              *
 *      process name, process arguments,               *
 *      environment variable, stack size (default),    *
 *      receive PID in child_pid, pass language,       *
 *      and specify not-an-orphan                      *
 *   Place the PID in the process array                *
 *                                                     *
 * ====================================================*/


void fork_task( void)
{
  u_int16 paths= 3;
  u_int32 edata= 0;
  int num;
  process_id child_pid;
  char orphan= 0;
  u_int16 type_lang= mktypelang( MT_PROGRAM, ML_OBJECT);

  argblk[1]= 0;
  for( num=0; num<ProcNb; num++) {
    argblk[0]= Proc[num].name;
    if( (errno= _os_exec( _os_fork, Proc[num].prior, paths, argblk[0],
			  argblk, _environ,
			  edata, &child_pid, type_lang, orphan)) != 0) {
      _errmsg( errno, "control : problem creating child %s\n", argblk[0]);
      fflush( stdout);
      sprintf( StrLog, "control:  problem creating child %s\n", argblk[0]);
      PrtLog( StrLog, 1);
      byebye( errno);
    }
    Proc[num].pid= child_pid;
    if( Debug >10) {
      printf( "control: fork %s Pid= %d\n", argblk[0], child_pid);
      fflush( stdout);
    }
  }
}

/* =================== AskBufMbx ======================*
 * Ask MsgSvrOut for a mailbox out buffer at pMessOut  *
 *   Wait if mailbox is full                           *
 *   Exit on other error                               *
 * ====================================================*/


void AskBufMbx( MailboxId mbx_out, int Bsize_out)
{
  error_code err;
  signal_code dummy;
  u_int32 t;

  while(( err= PostAlloc( mbx_out, (void **)&pMessOut, Bsize_out)) 
	!= E_SUCCESS) {
    if( err == POST_ERR_FULL) {
      t= 1; _os_sleep( &t, &dummy);         /* Mailbox full, try again */
      continue;
    }                                       /* Fatal error !!!!        */
    sprintf( StrLog, "control: Can't allocate a Mailbox buffer %s\n", 
	     PostError( err));
    PrtLog( StrLog, 1);
    byebye( err);
  }
}

/* =================== ConfigSetting =======================*
 * on message                                               *
 *    M_CONFIG_SET - set IsChangeDac, (more not impl.)      *
 *    M_FLASH_TO_CONFIG - run Flash2Config, set IsChangeDac *
 *    M_CONFIG_TO_FLASH - (not impl.)                       *
 * =========================================================*/


error_code ConfigSetting( unsigned char type)
{
  switch( type) {
  case M_CONFIG_SET:
    st->IsChangeDac= 1;    /* For monitor to write new values */
    break;
  case M_FLASH_TO_CONFIG:
    Flash2Config();
    st->IsChangeDac= 1;    /* For monitor to write new values */
    break;
  case M_CONFIG_TO_FLASH:
    break;
  }
  return 0;
}

/* =================== SendMessage ====================*
 * Send the message at pMessOut with correct size,     *
 *      type, and version, and shrink out mailbox      *
 *                                                     *
 * ====================================================*/

error_code SendMessage( MailboxId mbxid, int size,
			MsgTypeOut type, char version)
{
  error_code err;

  pMessOut->mheader.length= size+ sizeof(MESSAGE_HEADER);
  pMessOut->mheader.type=   type;
  pMessOut->mheader.version= version;
  if(( err= PostSendShrink( mbxid, pMessOut, size+sizeof(MESSAGE_HEADER))) 
     != E_SUCCESS) {
    sprintf( StrLog, "control: %s\n", PostError( err));
    PrtLog( StrLog, 1);
  }	
  fflush( stdout);
}

/* =================== Create_Event ===================*
 * Find the PID for process *name, and send sig to it  *
 *                                                     *
 * ====================================================*/


void SendSig( MSGSIG sig, char *name)
{
  error_code err;
  int n;

  for( n=0; n<ProcNb; n++) {
    if( strcmp( Proc[n].name, name)) continue;
    _os_send( Proc[n].pid, sig);
  }
}



/* ================= SendPrevLogFile ==================*
 * Send previous log file to Central Station           *
 *                                                     *
 * ====================================================*/

#define MAX_LOG_CHARS 100

int SendPrevLogFile( char *p)
{
  int size= 0, n;
  char str[100], *ps;
  static int num=0;
  static FILE *PrevLog= NULL;
  unsigned char comp;
  u_int32 *pcomp= (u_int32 *)p;

  p += 4;                            /* skip first 4 bytes */
  if( num == 0) {
    if(( PrevLog= fopen( PrevLogFileName, "r")) == NULL) {
      printf( "control : Can't open logfile %s\n", PrevLogFileName);
    }
  }
  if( PrevLog == NULL) return 1;
  while( fgets( str, 99, PrevLog)) {
    n=strlen( str); ps= str;
    size += n;
    while(n--) *p++= *ps++;
    if( size > MAX_LOG_CHARS) break;
  }
  if( size > MAX_LOG_CHARS) {
    if( num) comp= COMPLETION_NEXT;
    else     comp= COMPLETION_FIRST;
  }
  else {
    if( num) comp= COMPLETION_LAST;
    else     comp= COMPLETION_ALL;
  }
  num++;
  *pcomp= (comp << 24) | size;
  SendMessage( Pproc->mbx_out, size+4, M_LOG_SEND,0);
}

/* =================== Download  ======================*
 * Receive a file from Central Station                 *
 * Put it on Ram Disk                                  *
 * ====================================================*/


int Download( void)
{
  unsigned char *mesbuf;
  unsigned char *mesbufout= pMessOut->buf;
  int n, len, totlen;
  static int meslength=0;
  char fname[80];
  static FILE *fout= NULL;

  mesbuf= pMessIn->buf;
  switch(  pMessIn->mheader.completion) {
  case COMPLETION_FIRST:
    st->Freeze=1;
  case COMPLETION_ALL: 
    fout= NULL;
    n= *mesbuf++;                /* File name length */
    if( n > 70) {
      sprintf( StrLog, "control: download: bad file length name : %d\n",
	       n);
      PrtLog( StrLog, 1);
      *mesbufout= 0;
      SendMessage( Pproc->mbx_out, 1, M_DOWNLOAD_ACK, 0);
      return 1;
    }
    memcpy( fname, mesbuf, n);   /* File name */
    fname[n]= '\0';
    mesbuf += n;
    memcpy( &totlen, mesbuf, 4);   /* File name length */
    meslength= n+5;
    len= pMessIn->mheader.length- meslength;
    mesbuf += 4;
    *mesbufout++= n;             /* Save n, fname for message acknowledge */
    memcpy( mesbufout, fname, n);
                                 /* Open file in write mode */
    if( (fout= fopen( fname, "w")) == NULL) {
      PrtLog( "control: dowload: Can't open file in write mode\n", 1);
      return 1;
    }
    printf( "control: download: file %s: %d bytes\n", fname, totlen);
    fflush( stdout);
    break;
  case COMPLETION_NEXT: case COMPLETION_LAST:
    if( fout == NULL) {
      PrtLog( "control: dowload: something wrong occurs\n", 1);
      return 1;
    }
    len= pMessIn->mheader.length;
    break;
  }
  fwrite( mesbuf, len, 1, fout);
  if( pMessIn->mheader.completion == COMPLETION_ALL ||
      pMessIn->mheader.completion == COMPLETION_LAST) {
    fclose( fout);
    fout= NULL;
    SendMessage( Pproc->mbx_out, meslength, M_DOWNLOAD_ACK, 0);
    st->Freeze=0;
  }
}


/* ===================== OS9Cmd  ======================*
 * Receive an OS9 command from Central Station         *
 * Execute it                                          *
 * Send the command return code to the Central Station *
 * ====================================================*/


int OS9Cmd( void)
{
  unsigned char *mesbuf;
  unsigned char *mesbufout= pMessOut->buf;
  char cmd[150];
  int n, ret=0;

  mesbuf= pMessIn->buf;
  n= *mesbuf++;                /* command length */
  if( n > 140) {
    sprintf( StrLog, "control: OS9Cmd: bad command length : %d\n", n);
    PrtLog( StrLog, 1);
    ret= 1000;
  }
  else {
    memcpy( cmd, mesbuf, n);   /* command string */
    cmd[n]= '\0';
    ret= system( cmd);
    printf( "control: OS9cmd: %s\n", cmd); fflush( stdout);
  }
  memcpy( mesbufout, &ret, 4);
  SendMessage( Pproc->mbx_out, 4, M_OS9_CMD_ACK, 0);
}

int ReadyMessage( unsigned char *p)
{
  /* - 2 first bytes : hardware number of the Controller board
     - third byte : 
        - bit 0 : Set if current Config data module is equal to Config in 
	          FlahEprom
	- bit 7 : Set if RunEnable
     - Fourth byte :
        - bit 0 : unset if monitoring warning orange
	- bit 1 : unset if monitoring warning red
	- bit 7 : unset if GPS problem
	- all other bits are set
  */
  memcpy( p, &cf->AcqParams.HardAddress, 2);
  p += 2;
  *p++ = 0xFF;
  *p++ = 0xFF;
  return 4;
}

/*============== T1 management - to be changed ==============*/



void StartT1( void)
{
  static first= 1;
  if( first) {
    sleep( 6);    /* 28/10/99 */
    first= 0;
    if( dm_init( T1ModName, sizeof( TRIG1), (void **)&T1Data, &T1Header)) {
      printf( "***** Can't create %s data module (errno= %d) *****\n", 
	      T1ModName, errno); fflush( stdout);
      exit( (int)0);
    }
    t1= T1Data;
    t1->RunNumber= 0;
  }
  t1->IsTrig1= 1; 
  t1->rate= 10;
}


void StopT1( void)
{
  /*  t1->rate= 0; 
      t1->IsTrig1= 0; */
}



void dmp( char * title, unsigned char *p, int size)
{
  int n;

  printf( "Dump : %s - size= %d", title, size);
  for( n=0; n<size; n++) {
    if( !(n % 20)) printf( "\n");
    printf( "%3X", *p++);
  }
  printf( "\n");
  fflush( stdout);
}    

