/* 
   evtsvr : Local Station Software - Pierre Auger Project
   --------

              JM Brunet, L Gugliemi, C Le Gall, G Tristram
	               College de France - Paris

   Created    01/06/97 - JMB & CL
   Modified : 14/09/97 - Split MsgSvr into MsgSvrIn & MsgSvrOut
              17/11/98 - Version 2
	      03/03/99 - Version 2.1 : Search in MuBuffer
	      20/10/99 - Changes in Buffer management
	      26/10/99 - Add ALREADY message


----------------------------------------------------------------------------*/


#include <stdio.h>
#include <errno.h>

#define MAIN
#define MYNAME  EVTSVR


#include "msgdefs.h"
#include "dmlib.h"
#include "ready_acq.h"
#define  _STATUS_MAIN_
#include "status.h"

#include "augererr.h"
#include "evbdefs.h"


void init( void);
void DoWork( void);
void byebye( void);
int  SearchEvent( unsigned short evtid);
int  SendSlow( void *buf);
int  SearchSlow( unsigned int sec, unsigned int mindecanano, 
		unsigned int maxdecanano, SLOW_EVENT *ps, SLOW_EVENT *pr);
int  PutData( SLOW_MUON *psd, SLOW_MUON *prd, unsigned int min_tag, 
	     unsigned int max_tag, unsigned short nmuons, int flag);
int  CompareFast( void *pdata, unsigned char tag);
int  CompareSlow( void *pdata, unsigned char tag);
int  SearchAlready( void);
 
PROCESS *Pproc;

struct {
  int second, micro, isevent;
  unsigned short eventid;
} Already;

int         Debug=1, IsFirstBuffer;
signal_code Signal = 0 ;
TimeStamp   Wanted;
int         Status, Delta;
int SecondMin, MicroMin, SecondMax, MicroMax, IsAlready= 0;
char StrLog[256];

void
sig_hand( signal_code s )
{
  Signal= s;
  _os_rte();
}


main( int argc, char *argv)
{
  init();
  DoWork();
  sprintf( StrLog, "evtsvr: unexpected end\n");
  PrtLog( StrLog, 1);
  byebye();
}


void DoWork( void)
{
  signal_code dummy;
  u_int32 t;
  unsigned char *mesbuf;
  char microref;
  error_code err;
  int size_in;
  MESSAGE *pMessIn;
  unsigned short evid;

  while( 1) {                           /*****   Start infinite loop  *****/
    t=0; _os_sleep( &t, &dummy );
    if( Signal == SIGQUIT) byebye();
                                        /* Read a message from Mailbox-in */
    while((err= PostReceive( Pproc->mbx_in, (void **)&pMessIn, &size_in))
       == E_SUCCESS) {
      if( Signal != Pproc->notify->param1) {
	printf( "evtsvr : Unknown signal %d received\n", Signal);
	fflush( stdout);
	sprintf( StrLog, "evtsvr: Unknown signal %d received\n", Signal);
	PrtLog( StrLog, 1);
	continue;
      }
      if( pMessIn->mheader.type == M_T3_YES) {
	StData->evtsvr.T3Asked++;      /* CS asks EvtSvr to give it an Event */
	mesbuf= pMessIn->buf;
	evid= *(unsigned short *)mesbuf; mesbuf += 2;
	Wanted= *(TimeStamp *)mesbuf; 
	StData->evtsvr.second= Wanted.second;
	mesbuf += sizeof( TimeStamp);
	                           /* Station position in a list */
	mesbuf += (pMessIn->mheader.completion & 0x3f) << 1;
	if( *mesbuf & 0x80) microref= -(*mesbuf++ & 0x7F);
	else microref= *mesbuf++;
        Wanted.nano += microref;
	Delta= *mesbuf++;
                                   /* Min & Max timeStamp requiered */
	SecondMin= SecondMax= Wanted.second;
	if( Delta > Wanted.nano) {
	  SecondMin--;
	  MicroMin= 1000000- (Delta- Wanted.nano);
	}
	else MicroMin= Wanted.nano- Delta;
	if( Wanted.nano + Delta > 1000000) {
	  SecondMax++;
	  MicroMax= Wanted.nano+ Delta- 1000000;
	}
	else MicroMax= Wanted.nano+ Delta;
	Status= SearchEvent( evid);
      }
      else {                      /* Unknown type of message */
	sprintf( StrLog,"evtsvr : unknown Message type (%d)\n", 
		   pMessIn->mheader.type);
	PrtLog( StrLog, 1);
      }
      PostCancel( Pproc->mbx_in, pMessIn);
    }                             /* End while( PostReceive()) */
    if( err == POST_ERR_EMPTY) continue;
    sprintf( StrLog, "evtsvr: %s\n        Last signal received: %d", 
	     PostError( err), Signal);
    PrtLog( StrLog, 1);
    byebye( );
  }                               /* End of infinite loop */
}


int CompareFast( void *pdata, unsigned char tag)
{
  FAST_EVENT *pfast;
  int second, micro;
  TimeStamp   ts;

  Status= M_T3_TOO_YOUNG;
  pfast= (FAST_EVENT *)pdata;
  second= pfast->datep.second;
  micro=  pfast->datep.nano;
  if( second > SecondMax) {
    if( IsFirstBuffer) Status= M_T3_LOST;
    else               Status= M_T3_NOT_FOUND;
    return M_T3_NOT_FOUND;
  }
  IsFirstBuffer= 0;
  if( second  < SecondMin) return EVB_ERR_NOT_FOUND;
  if( second == SecondMin && micro < MicroMin) return EVB_ERR_NOT_FOUND;
  if( second == SecondMax && micro > MicroMax) {
    Status= M_T3_NOT_FOUND;
    return  M_T3_NOT_FOUND;
  }
  Status= NO_ERROR;
  return SUCCESS;
}

int CompareSlow( void *pdata, unsigned char tag)
{
  SLOW_EVENT *pslow;
  TimeStamp   ts;

  Status= M_T3_NOT_FOUND;
  pslow= (SLOW_EVENT *)pdata;
  return EVB_ERR_NOT_FOUND;
}


int SearchEvent( unsigned short evtid)
{
  int size_out, size;
  unsigned short *ErrorWord;
  signal_code dummy;
  u_int32 t;
  unsigned char *mesbuf;
  error_code err;
  MESSAGE *pMessOut;
  MsgTypeOut type;
  int itag=0;              /* index in the tag table */
  unsigned char tag=0;     /* requested tag value    */
  unsigned char newtag=1;  /* new tag value          */

                                            /* Ask for a MailBox */
  size_out= Pproc->Bsize_out;
  while(( err= PostAlloc( Pproc->mbx_out, (void **)&pMessOut, size_out)) 
	!= E_SUCCESS) {
    if( err == POST_ERR_FULL) {
      t= 1; _os_sleep( &t, &dummy);         /* Mailbox full, try again */
    }
    else {
      sprintf( StrLog,"evtsvr: PostAlloc error %d\n",err);
      PrtLog( StrLog, 1);
    }
  }
  mesbuf = pMessOut->buf;
  *(unsigned short *)mesbuf= evtid;    mesbuf += sizeof( short);
  ErrorWord= (unsigned short *)mesbuf; mesbuf += sizeof( short);
  *ErrorWord= NO_ERROR;
  size=sizeof(FAST_EVENT);
  type=  M_T3_EVT;
  if( !SearchAlready()) {
                                            /* Try to find the Event */
    IsFirstBuffer= 1;
    EvbFindTag( evbid, (void *)mesbuf, &size, CompareFast, itag, tag, newtag);
    if( 0 && Status != NO_ERROR) {         /* Not Found -> Search for muons */
      if( ( size=SendSlow( (void *)mesbuf))) {
	type=  M_T3_MUONS;
	Status= NO_ERROR;
      }
    }
  }
  switch( Status ) {
  case NO_ERROR:
    Already.second= Wanted.second;
    Already.micro=  Wanted.nano;
    Already.eventid= evtid;
    if( type= M_T3_EVT) Already.isevent= 1;
    else                Already.isevent= 0;
    IsAlready=1;
    StData->evtsvr.T3Found++;
    break;
  case M_T3_NOT_FOUND:
    StData->evtsvr.T3NotFound++;
    *ErrorWord= M_T3_NOT_FOUND;
    size= 0; break;
  case M_T3_LOST:
    StData->evtsvr.T3Lost++;
    *ErrorWord= M_T3_LOST;
    size= 0; break;
  case M_T3_TOO_YOUNG:
    StData->evtsvr.T3TooYoung++;
    *ErrorWord= M_T3_TOO_YOUNG;
    size= 0; break;
  case M_T3_ALREADY:
    StData->evtsvr.T3Already++;
    *ErrorWord= M_T3_ALREADY;
    if( Already.isevent) type= M_T3_EVT;
    else                 type= M_T3_MUONS;
    *(unsigned short *)mesbuf= Already.eventid;
    size= sizeof( Already.eventid);
    break;
  default:
    StData->evtsvr.T3NotFound++;
    *ErrorWord= M_T3_NOT_FOUND;
    size= 0; break;
  }
  size += sizeof(MESSAGE_HEADER)+ sizeof( int);
  pMessOut->mheader.length= size;
  pMessOut->mheader.type=  type;
  if(( err= PostSendShrink( Pproc->mbx_out, pMessOut, size))
     != E_SUCCESS) {
    sprintf( StrLog,"evtsvr: %s\n", PostError( err));
    PrtLog( StrLog, 1);
  }
  return Status;
}


int SearchAlready()
{
  /* Search if an event, in the time range given, has already been sent
     to the Central Station.         */
  if( !IsAlready) return 0;          /* No event sent yet */
  if( Already.second  < SecondMin) return 0;
  if( Already.second == SecondMin && 
      Already.micro < MicroMin) return 0;
  if( Already.second == SecondMax && 
      Already.micro > MicroMax) return 0;
  Status= M_T3_ALREADY;
  return 1;
}

int SendSlow( void *mesbuf)
{
  int size_in, size, Status, *ErrorWord;
  signal_code dummy;
  u_int32 t;
  error_code err;
  MESSAGE *pMessOut;
  MsgTypeOut type;
  static SLOW_EVENT SlowEvent;
  unsigned int sec, mindecanano, maxdecanano;
  int itag=0;            /* index in the tag table */
  unsigned char tag;     /* requested tag value    */
  unsigned char newtag;  /* new tag value          */
  
  Wanted.nano *= 100;                      /* micro to deca-nano */
  Delta       *= 100;
  sec= Wanted.second;
  mindecanano= Wanted.nano- Delta;
  maxdecanano= Wanted.nano+ Delta;
printf( "   SendSlow: sec= $%x - min= $%x - max= $%x\n", sec,
mindecanano, maxdecanano); fflush( stdout);
  size_in=sizeof( SLOW_EVENT)-100;
  type=  M_T3_MUONS;
  size= 0;
                                            /* Try to find the Event */
printf( "   Just before EvbFind\n"); fflush( stdout);
printf( "   mubid= %d - size_in= %d - sec= $%x - nano= $%x\n",
mubid, size_in, Wanted.second, Wanted.nano); 
printf( "   SlowEvent size= %d\n", sizeof( SlowEvent));
fflush( stdout);
  Delta= 10000000;
  if((err = EvbFindTag( mubid, (void *)mesbuf, &size, CompareSlow, itag, tag,
			  newtag))) {
    sprintf( StrLog,"evtsvr: error %d in  EvbFind (SendSlow function)\n", 
	       err);
    PrtLog( StrLog, 1);
    return 0;
  }
printf( "   Just after EvbFind satus= %d\n", Status); fflush( stdout);
  if( Status == NO_ERROR) {              /* Select muons from Time Range */
    if( !( size= SearchSlow( sec, mindecanano, maxdecanano, &SlowEvent,
			     (SLOW_EVENT *)mesbuf))) return 0;
    StData->evtsvr.T3Muon++;
  }
else printf( "evtsvr: Event not found in Muon Buffer\n");
  return size;
}


int SearchSlow( unsigned int sec, unsigned int mindecanano, 
		unsigned int maxdecanano, SLOW_EVENT *ps, SLOW_EVENT *pr)
{
  /* Depending on the value of the local variable flag, there are 3 different
     cases :
     - flag= 0 : standard case : all the data in the buffer are sorted in time
                 and the requiered range is entirely contained in the buffer.
     - flag= 1 : The data are still sorted, but the range extend to the next 
                 buffer. A call to FindNextEvent is then requiered to fill the
		 last part of the ResBuffer with part of the New MuonBuffer.
     - flag= 2 : The 25 MHz clock counter (24 bits wide) is cleared in the 
                 middle of the current MuonBuffer. So there are 2 ranges of 
		 TimeTag to take into account.
     One must take care of a buffer containing data from two different
     seconds.
  */
  SLOW_MUON  *psd= ps->data;
  SLOW_MUON  *prd= pr->data;
  int size_in, size;
  error_code err;
  unsigned short nmuons;
  unsigned int min_tag, max_tag;
  int mini;
  int nres= 0, fmr;
  int flag=0;
  int itag=0;            /* index in the tag table */
  unsigned char tag;     /* requested tag value    */
  unsigned char newtag;  /* new tag value          */

printf( "   SearchSlow: sec= $%x - min= $%x - max= $%x\n", sec,
mindecanano, maxdecanano); fflush( stdout);
printf( "               sec in buffer= $%x - nano= $%x\n", 
	ps->date.second, ps->date.nano);
  if( ps->date.second != sec) {        /* 2 diff. seconds in the same buffer */
    if( (ps->date.second - 1) != sec ) return 0;
    if( ps->date.nano > 20000000) return 0;
    ps->date.second= sec;
    ps->date.nano= 100000000+ ps->date.nano;
  }
  if( maxdecanano > ps->date.nano) flag=1;    /* flag = 1 */
  mini= ps->date.nano- 10000000;
  if( mini < 0) mini= 0;
  if( mindecanano < mini || mindecanano > ps->date.nano) return 0;
  nmuons= ps->nmuons/(sizeof( SLOW_MUON)/sizeof( unsigned int));
  pr->nmuons= 0;
                                         /* From deca nanosecond to time_tag */
  fmr= ps->date.nano -maxdecanano;
  max_tag= ps->time_tag - ((fmr << 2)/10);
  fmr= ps->date.nano -mindecanano;
  min_tag= ps->time_tag - ((fmr << 2)/10);
  if( min_tag > max_tag) {               /*  flag = 2 */
    flag= 2;
    min_tag= (min_tag & 0xFFFFFF);
  }
                            /* Store relevant data ( from min_tag to max_tag */
printf( "evtsvr: min_tag=$%x - max_tag= $%X - nmuons= %d - flag= %d\n",
min_tag, max_tag, nmuons, flag);
  if( (nres= PutData( psd, prd, min_tag, max_tag, nmuons, flag))) {
    pr->date.second=  ps->date.second;
    pr->date.nano=    ps->date.nano;
    pr->time_tag=     ps->time_tag;
    pr->nmuons= nres*(sizeof( SLOW_MUON)/sizeof( unsigned int));
  }
                            /* flag = 1 ---> Call FindNextEvent() */
  if( flag != 1 || !nres) return( nres);
  prd += nres;
                            /* Look to next buffer */
  Delta= (Delta << 1);
  if((err = EvbFindTag( mubid, (void *)ps, &size_in, CompareSlow, itag, tag,
			  newtag))) {
    sprintf( StrLog, "evtsvr: error %d in SendSlow function\n", err);
    PrtLog( StrLog, 1);
    Delta= (Delta >> 1);
    return 0;
  }
  Delta= (Delta >> 1);
  if( Status == NO_ERROR) {  
    psd= ps->data;
    nmuons= ps->nmuons/(sizeof( SLOW_MUON)/sizeof( unsigned int));
    nres += PutData( psd, prd, min_tag, max_tag, nmuons, flag); 
    pr->nmuons= nres*(sizeof( SLOW_MUON)/sizeof( unsigned int));
  }
  return( nres*sizeof( SLOW_MUON));
}


int PutData( SLOW_MUON *psd, SLOW_MUON *prd, unsigned int min_tag, 
	     unsigned int max_tag, unsigned short nmuons, int flag)
{
  int n, nres=0, fl=0;
  unsigned int time_tag;
                       /* 2 time ranges : the 25 MHz counter go through
			  zero in the middle of the buffer */
  if( max_tag < 0x1000000) {        /* requiered range contains zero */
    if( flag == 2) for( n=0 ; n< nmuons; n++, psd++) {
      if( psd->time_tag > min_tag  || psd->time_tag < max_tag) {
	nres++;
	*prd++ = *psd;
      }
    }                               /* requiered range outside zero */
    else for( n=0 ; n< nmuons; n++, psd++) {
      if( psd->time_tag < min_tag) continue;
      if( psd->time_tag > max_tag) continue;
      nres++;
      *prd++ = *psd;
    }
  }
                     /* Standard case : all the data in the buffer are sorted
			in time and the requiered range is all contained in
			the buffer */
  else {
    for( n=0 ; n< nmuons; n++, psd++) {
      time_tag= psd->time_tag;
      if( time_tag < 0xB00000) {
	fl=1; 
	time_tag += 0x1000000;
      }
      if( time_tag < min_tag) continue;
      if( time_tag > max_tag) break;
      nres++;
      *prd = *psd;
      if( fl) prd->time_tag += 0x1000000;
      prd++;
    }
  }
  return nres;
}



void init( void)
{
  error_code err;
  signal_code sig;


                                   /* Link to Data Modules        */
  if( dm_init( StatModName, sizeof( STATUS), (void **)&StData, &StHeader)) {
    printf( "Can't link to %s data module (errno= %d)\n", 
			 StatModName, errno);
    exit( (int)0);
  }
  if(( Pproc= myprocess( MYNAME)) == NULL) byebye( );
                                   /* Open Mailbox in read mode */
  if(( err= ReadyInit( &ReadyEvId, ReadyEvName))) {
    sprintf( StrLog, "evtsvr: Error %d calling ReadyInit\n", err);
    PrtLog( StrLog, 1);
    byebye();
  }
  if((err= PostCreate( &Pproc->mbx_in, Pproc->pnamein, Pproc->Msize_in,
                       Pproc->notify))) {
    sprintf( StrLog, "evtsvr: Can't Create Mailbox \"%s\" for Rec. : %d\n", 
	       Pproc->pnamein, err ) ;
    PrtLog( StrLog, 1);
    byebye();
  }
                                   /* Event & Muon Buffers initialisation */
  while(( err = EvbInit(&evbid, EVT_BUFFER_NAME, EVBSIZE)) == E_KWNMOD);
  if( err) {
    sprintf( StrLog, "evtsvr: Error %d EvbInit for Event Buffer\n", err);
    PrtLog( StrLog, 1);
    byebye();
  }
  while(( err = EvbInit(&mubid, MU_BUFFER_NAME, MUBSIZE)) == E_KWNMOD);
  if( err) {
    sprintf( StrLog, "evtsvr: Error %d EvbInit for Muon Buffer\n", err);
    PrtLog( StrLog, 1);
    byebye();
  }
printf( "evtsvr : Init MuBuffer: mubid=%d\n", mubid); fflush( stdout);
  if(( err= ReadyToGo( ReadyEvId, READY_EVTSVR))) {
    sprintf( StrLog, "evtsvr: Error when calling ReadyToGo() : %d\n", err);
    PrtLog( StrLog, 1);
    byebye();
  }
  if(( err= ReadyAll( ReadyEvId, &sig, READY_MSGSVROUT)) != E_SUCCESS) {
    sprintf( StrLog, "Unexpected signal %d calling ReadyAll() : err= %d\n", 
	     sig, err);
    PrtLog( StrLog, 1);
    byebye();
  }
                                   /* Open Mailbox in send mode */
  if ( (err = PostLink( &Pproc->mbx_out, Pproc->pnameout ))) {
    sprintf( StrLog, "evtsvr: Can't Open Mailbox \"%s\" for Send : %d\n",
            Pproc->pnameout , err ) ;
    PrtLog( StrLog, 1);
    byebye();
  }
                                   /* Intercept Signals */
  _os_intercept( sig_hand, _glob_data );
  err= ReadyFinish( ReadyEvId, ReadyEvName);
  printf( "evtsvr : init ended...\n");
  fflush( stdout);
}



void byebye( void)
{
  error_code err;
                                              /* Unlink Status data module */
  printf( "evtsvr: Signal= %d\n", Signal);
  fflush( stdout);
  dm_finish( &StHeader );
                                              /* Event Buffers End */
  if (err=EvbFinish(&evbid))
    printf("evtsvr : Error %d in EvbFinish (evbid)\n", err);
  if (err=EvbFinish(&mubid))
    printf("evtsvr : Error %d in EvbFinish (mubid)\n", err);
                                              /* Close Mailboxes */
  PostClose( &Pproc->mbx_in);
  PostClose( &Pproc->mbx_out);
  exit( (int) errno);
}

