/*=======================================================================
  Edition History:                                                      
                                                                        
    Ed.   Date      What happened                                       
    --  --------    --------------------------------------------------- 
    01  95/05/11    Created                                             
    02  96/09/18    Added Stack Check Note                          rkw 
    03  97/01/10    Added PPC -bepg Note                            rkw 
    04  97/01/16    Added E_NOT me + headers ++                     rkw 
    05  97/03/11    Edited                                          jrt
    06  98/01/08    Adaptation au trigger Auger sur PPC403          LG

When started, the interrupt is disabled.
Upon reception of the signal 1000, the it is enabled.
Upon reception of the signal 1001, the it is disabled.

=======================================================================

A System State Process which installs an Interrupt Handler


This is a short program that installs an interrupt routine on a vector
and prints a message when an interrupt occurs. It could be  
compilied with: cc irqexample.c -r -bepg -tp=ppc -l=cpu.l -l=sys_clib.l,
you must define the vector # for PPC and and add in the correct -tp for
your processor.
*/

/*--------------------------------------------------------------------------!
! NOTE: This process must be System State!  To compile as a system state    !
!	process you can either specify 'syscstart.r' as the first file on   !
!	the 'link line' in your makefile or if your compiling directly      !
!	with 'cc' you must include the option: '-cs=syscstart.r' on your    !
!	command line. You must also add the -r option to disable stack      !
!       checking. If your are using a machine which does not have direct    !
!       access to the <pc> such as PowerPC's the -bepg option is necessary. !
!	This does not use a data module to display the IRQ's.  This is also !
!	CPU time consuming, as it prints out the number of IRQ's whether    !
!	more have occurred or not.                                          !
!--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------!
!                       extirq <num> <edge> <sign> <event-name>             !
!                                                                           !
!   num    : External interrupt number ( 0 to 4 )                           !
!   edge   : 1 = Edge sensitive, 0 = level senesitiv                        !
!   sign   : 0 = Negative level/transition, 1 = positiv level/transition    !
!   event  : Name of the event to signal ( by "_os_ev_signal" )             !
!                                                                           !
!                                                                          */

#include <types.h>
#include <const.h>
#include <errno.h>
#include <cglob.h>
#include <signal.h>
#include <events.h>

#include "gpsirq.h"
#define ENABLE_SIG 1000
#define DISABLE_SIG 1001

static signal_code Sig = 0 ;

extern void             *_glob_data;

/*------------------------------------------------------------------!
! irqfunc: This is the function that is run by the Kernel when the  !
!	   interrupt fires off.  This function MUST access the      !
!	   hardware to 'clear down' the interrupt.                  !
!------------------------------------------------------------------*/
irqfunc()
{        
  int dummy ;

  /* Check the Interrupt status and see if it is my interrupt */

  if ( (getexisr() & IrqEnable) && (getexier() & IrqEnable) ) {
    /* It's me */
    /* Clear the interrupt bit in EXISR */
    setexisr( IrqEnable ) ;
    /* Signal the Event */
    _os_ev_signal( IrqEvId, &dummy, 1 ) ;
    return SUCCESS; 
  }
  else return E_NOTME;
}

void
sig_hand( signal_code sig )
{
  Sig = sig ;
  _os_rte() ;
}

help()
{
  puts( "Syntax  : extirq <opts>" ) ;
  puts( "Function: Sets a function on an external interrupt ( PPC 403 )" ) ;
  puts( "Options :" ) ;
  puts( "\t-n[=]<num>   : Interrupt number ( 0 to 4 )" ) ;
  puts( "\t-p[=]<prior> : Polling Priority" ) ;
  puts( "\t-a[=]<port>  : Hardware Port Address" ) ;
  puts( "\t-s[=]<event> : Name of the event to be signaled" ) ;
  puts( "ALL the previous options are mandatory !" ) ;
  puts( "\t-lp          : Positive Level" ) ;
  puts( "\t-ln          : Negative Level" ) ;
  puts( "\t-ep          : Positive Edge" ) ;
  puts( "\t-en          : Negative Edge" ) ;
  puts( "ONE ( and only one ) of the 4 previous options is MANDATORY !" ) ;
  exit( 1 ) ;
}

/*--------------------------------------------------------------------------!
! main: This function sets up the interrupt and signals an event at each    !
!       occurence of the Interrupt.                                         !
!--------------------------------------------------------------------------*/
main( int argc, char **argv )
{
  int loop ;
  u_int16
    vector,			/* interrupt vector number */
    priority;	/* interrupt priority (0 - 255)	*/
  error_code	error;

  /* Get the parameters */

  argv++ ;
  for( ; *argv ; argv++ ) {
    char *p = *argv ;
    if ( *p != '-' ) help() ;
    p++ ;
    switch( *p ) {
    case 'n': p++ ; if ( *p == '=' ) p++ ;
      sscanf( p, "%d", &IrqNum ) ;
      break ;
    case 'l': p++ ;
      LevelEdge = LEVEL ;
      if ( *p == 'p' ) Sign = POSITIVE ;
      else Sign = NEGATIVE ;
      break ;
    case 'e': p++ ;
      LevelEdge = EDGE ;
      if ( *p == 'p' ) Sign = POSITIVE ;
      else Sign = NEGATIVE ;
      break ;
    case 'a': p++ ; if ( *p == '=' ) p++ ;
      sscanf( p, "%x", &IrqPort ) ;
      break ;
    case 'p': p++ ; if ( *p == '=' ) p++ ;
      sscanf( p, "%d", &IrqPrior ) ;
      break ;
    case 's': p++ ; if ( *p == '=' ) p++ ;
      IrqEvName = p ;
      break ;
    default: help() ;
    }
  }
  if( IrqNum == -1 || LevelEdge == -1 || Sign == -1 || IrqEvName == NULL )
    help() ;

  /* Prepare IOCR, EXIER and EXISR */
  IrqIocr = 0 ;
  IrqIocrMask = IntIocrMask[IrqNum] ;
  if ( LevelEdge == EDGE ) IrqIocr |= IntEdge[IrqNum] ; 
  if ( Sign == POSITIVE ) IrqIocr |= IntSign[IrqNum] ;
  IrqEnable = IntEnable[IrqNum] ;
  IrqDisable = IntDisable[IrqNum] ;

  /* Install signal handler */
  _os_intercept( sig_hand, _glob_data ) ;

  /* Initialize necessary parameters */
  vector = EXT_VECTOR ;
  priority = IrqPrior ;

  /* Install the interrupt service function	*/
  if ((error = _os_irq(
		       vector,
		       priority,
		       irqfunc,
		       _glob_data ))
      != SUCCESS)
    exit(_errmsg(error, "Can't install interrupt service function!\n"));

  /* Link to or create an Event */
  if ( _os_ev_creat( -1, 1, 0x777, &IrqEvId,
		     IrqEvName, 0, 0 ) != SUCCESS &&
       _os_ev_link( IrqEvName, &IrqEvId ) != SUCCESS ) {
    printf( "Can't Create or Link to the event!\n") ;
    exit( 1 ) ;
  }
  
  /* Disable Interrupts */
  setiocr( (getiocr() & IrqIocrMask) | IrqIocr ) ;
  setexisr( getexisr() | IrqEnable ) ; /* Clear ISR */
  setexier( getexier() | IrqEnable ) ; /* Enable IRQ's */

  /* Infinite loop. Ends upon any signal */
  for( ; ; ) {
    unsigned int tt = 0 ;
    signal_code dummy ;

    tt = 0 ;
    dummy = 0 ;
    _os_sleep( &tt, &dummy ) ;
    if ( dummy == ENABLE_SIG ) {
      setexier( getexier() | IrqEnable ) ;
      setexisr( getexisr() | IrqEnable ) ;
    }     
    else if ( dummy == DISABLE_SIG )
      setexier( getexier() & IrqDisable ) ;
    else if( dummy ) break ;
  }

  /* Disable Interrupts and clear */
  setexier( getexier() & IrqDisable ) ;
  setexisr( getexisr() | IrqEnable ) ;

  /* Deinstall IRQs */
  _os_irq(
	  vector,
	  priority,
	  NULL,
	  _glob_data ) ;

  /* Unlink from Event */
  _os_ev_unlink( IrqEvId ) ;
  exit( 0 ) ;
}

