/* 
   monitor : Local Station Software - Pierre Auger Project
   -------
   Created    08/09/99 - JMB
   Modified : 23/05/00 - JMB : Version 2.4
----------------------------------------------------------------------------*/
#include <alarm.h>
#include <errno.h>

#define _STATUS_MAIN_
#include "status.h"
#define _MONITOR_MAIN_
#define _CONFIG_MAIN_
#include "run_config.h"
#include "ready_acq.h"
#include "augererr.h"
#include "evblib.h"

#define SIG_ALARM  14   /* Signal for Cycle Alarm */
void init( void);
void byebye( error_code);
void InitSave( void);
void AddDacs( MONITOR_WRITE *pw);
void WriteMonitor( void);


signal_code Signal = 0 ;

char StrLog[256];
unsigned short SaveDac[NDAC], SaveOutputRegister;

void
sig_hand( signal_code s )
{
  Signal= s;
  _os_rte();
}


main( int argc, char *argv)
{
  unsigned int nt=2;
  signal_code dummy;
  unsigned short AdcChan[NADC], chan[NADC], *p;
  MONITOR *pmean;
  unsigned int sec;
  error_code err;
  int n, number,wred, worange;
  unsigned char tag[]= {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};

  init();
  number=0;
  for( n=0; n<NADC; n++) AdcChan[n]= 0;
  while(1) {
    nt= 0; _os_sleep( &nt, &dummy);   /* Wait for an Alarm */
    if( Signal == SIGQUIT) byebye( 2000);
    if( StData->RunStatus == REBOOT) byebye(2001);
    if( StData->IsChangeDac) {        /* Write some DAC values */
      StData->IsChangeDac=0;
      WriteMonitor();
    }
                                      /* next line must be changed */
    if( !number) sec= StData->inter.SlowSec;
                                     /* Read ADC & increment */
    p= chan;
#ifdef FAKE_OLEG
    ReadADC( p, CfData->MonitParams.ADCinUSE,
             CfData->MonitParams.IsTPC);
#else
    ReadADC( p, CfData->MonitParams.ADCinUSE);
#endif
    for( n=0; n<NADC; n++) AdcChan[n] += chan[n];
    if( ++number < CfData->MonitParams.Times) continue;
    for( n=0; n<NADC; n++) AdcChan[n] /= CfData->MonitParams.Times;
                                      /* Event Buffer Allocation */

    while( err = EvbAlloc( MonitId, (void **)&pmean, sizeof(MONITOR))) {
      if( err != EVB_ERR_FULL) {
        sprintf( StrLog, "monitor: error $%x in EvbAlloc\n",err);
	PrtLog( StrLog, 1);
        byebye( err);
      }
      
      if( err = EvbFree( MonitId, NB_MONIT_RELEASE)) {
        sprintf( StrLog, "monitor: EvbFree: err= $%X\n",err);
	PrtLog( StrLog, 1);
        byebye( err);
      }
    }
    
    pmean->Second= sec;              /* Write Monitoring Buffer */
    pmean->ItemNb= number;
    pmean->Summary= StData->GpsState << 16;
    if( !CfData->AcqParams.RunEnable) pmean->Summary |= 0x80000000;
    wred= worange=0;
    for( n=0; n<NADC; n++) {
      if( !CfData->MonitParams.ADCinUSE) { AdcChan[n] |= 0x8000; continue; }
      if( (AdcChan[n] < CfData->MonitParams.MinRed[n]) ||
          (AdcChan[n] > CfData->MonitParams.MaxRed[n])) {
	wred++; AdcChan[n] |= 0x4000; continue;
      }
      if( (AdcChan[n] >= CfData->MonitParams.MinOrange[n]) &&
          (AdcChan[n] <= CfData->MonitParams.MaxOrange[n])) continue;
      worange++; AdcChan[n] |= 0x2000;
    }
    if( wred) pmean->Summary |= 1;   /* Warning red or orange */
    else if( worange) pmean->Summary |= 2;
    for( n=0; n<NADC; n++) pmean->Read.Chan[n]= AdcChan[n];
    p= pmean->Write.Chan;
#ifdef FAKE_OLEG
    ReadDAC( p, CfData->MonitParams.DACinUSE,
             CfData->MonitParams.IsTPC);
    ReadRegister( &pmean->Write.OutputRegister, 0);
    ReadRegister( &pmean->Read.InputRegister, 1);
#else
    ReadDAC( p, CfData->MonitParams.DACinUSE);
    pmean->Write.OutputRegister= ReadRegister();
    pmean->Read.InputRegister= ReadBackRegister();
#endif
                                     /* Mark MonitBuffer as ready */
    if( err= EvbReady( MonitId, (void *)pmean, sizeof(MONITOR),tag)) {
      sprintf( StrLog, "monitor: EvbReady: err= %d\n", err);
      PrtLog( StrLog, 1);
      break;
    }
    number=0;                        /* Reset ADC values */
    for( n=0; n<NADC; n++) AdcChan[n]= 0;
    Signal= 0;
                                     /* Send signal to CalMonSvr */
    if( CfData->MonitParams.IsSendRegular && !StData->Freeze) {
      kill( StData->calmon.PidCalMon, SIG_MONITOR);
    }
  }
  byebye( (error_code) 0);
}


void init( void)
{
  error_code err;
  alarm_id AlarmId;
  u_int32 tm;

                                      /* Link to Data Modules        */
  if( dm_init( StatModName, sizeof( STATUS), (void **)&StData, &StHeader)) {
    printf( "monitor: ***** Can't create %s data module (errno= %d) *****\n", 
      StatModName, errno);
    exit( (int)0);
  }
  
  if( dm_init( ConfigModName, sizeof( CONFIG), 
               (void **)&CfData, &CfHeader)) {
    sprintf( StrLog, "monitor: Can't create %s DM (errno= %d)\n",
	     ConfigModName, errno);
    PrtLog( StrLog, 1);
    exit( (int)0);
  }
 

  InitSave();                         /* Init DAC & OutputRegister values */
                                      /* Monit Buffer initialisation */
  while(( err= EvbInit(&MonitId, MONIT_BUFFER_NAME, MONIT_SIZE)) == E_KWNMOD );
  if( err) {
    sprintf( StrLog, "monitor: EvbInit: err= $%x\n", err);
    PrtLog( StrLog, 1);
    byebye( err);
  }
  
                                      /* Intercept Signals */
  _os_intercept( sig_hand, _glob_data );
                                      /* Set a Cycle Alarm */
  tm= 0x80000000 + CfData->MonitParams.ReadRate*256;
  if(( err= _os_alarm_cycle( &AlarmId, SIG_ALARM, tm))) {
    sprintf( StrLog, "monitor : _os_alarm_cycle(): err= %d\n",err);
    PrtLog( StrLog, 1);
  }
  WriteMonitor();                     /* Write DAC and Output Register */
  printf( "monitor : init ended...\n"); fflush( stdout);
}



void InitSave( void)  /* Set SaveOutputRegister & active Dac[]
                         to Config values -1                   */
{
  int i;
  unsigned char m=1;
  MONIT_PARAMS  *pconfig= &CfData->MonitParams;
  
  for( i=0; i<NDAC; i++) {
    if( !(m & pconfig->DACinUSE)) SaveDac[i]= 0;
    else SaveDac[i]= pconfig->Dac[i]-1;
    m <<=1;
  }
  SaveOutputRegister= pconfig->OutputRegister-1;
}


void WriteMonitor( void)   /* Write SaveOutputRegister & active Dac[] 
                              that have change from the previous write */
{
  int i;
  unsigned char m=1;
  unsigned char InUse=0;
  MONIT_PARAMS  *pconfig= &CfData->MonitParams;
 
  if( SaveOutputRegister != pconfig->OutputRegister) {
    SaveOutputRegister= pconfig->OutputRegister;
    WriteRegister( SaveOutputRegister);
  }
  for( i=0; i<NDAC; i++) {
    if(( m & pconfig->DACinUSE) && ( SaveDac[i] != pconfig->Dac[i])) {
      InUse |= m;
      SaveDac[i] = pconfig->Dac[i];
    }
    m <<=1;
  }
  if( InUse) {
#ifdef FAKE_OLEG
    WriteDAC( SaveDac, InUse, pconfig->IsTPC);
#else
    WriteDAC( SaveDac, InUse);
#endif
  }
}


void AddDacs( MONITOR_WRITE *pw)   /* Add DAC values to the buffer */
{
  MONIT_PARAMS  *pconfig= &CfData->MonitParams;
  int i;

  for( i=0; i<8; i++) pw->Chan[i]= pconfig->Dac[i];
  pw->OutputRegister= pconfig->OutputRegister;
}


void byebye( error_code err)
{
  dm_finish( &StHeader );
  dm_finish( &CfHeader );
  EvbReset( MonitId);
  exit( 0);
}

#ifdef FAKE_OLEG
/*----------------- Fake functions : take Oleg's ones --------------*/

unsigned short FakeADC[]= { 
  0x100, 0x101, 0x102, 0x103, 0x104, 0x105, 0x106, 0x107,
  0x200, 0x201, 0x202, 0x203, 0x204, 0x205, 0x206, 0x207,
  0x300, 0x301, 0x302, 0x303, 0x304, 0x305, 0x306, 0x307,
  0x400, 0x401, 0x402, 0x403, 0x404, 0x405, 0x406, 0x407
};

unsigned short FakeDAC[]= {
  0x500, 0x501, 0x502, 0x503, 0x504, 0x505, 0x506, 0x507
};

unsigned short FakeInReg= 0xCAFE, FakeOutReg= 0xFADE;

error_code ReadADC( unsigned short *pchan, unsigned int InUse,
                    int IsTPC)
{
  int n;
  unsigned int k=1;

  for( n=0; n<NADC; n++) {
    if( InUse & k) *pchan= FakeADC[n];
    else *pchan= 0;
    pchan++;
    k <<= 1;
  }
  return 0;
}




error_code ReadDAC( unsigned short *pchan, unsigned char InUse,
                    int IsTPC)
{
  int n;
  unsigned char k=1;

  for( n=0; n<NDAC; n++) {
    if( InUse & k) *pchan= FakeDAC[n];
    else *pchan=0;
    pchan++;
    k <<= 1;
  }
  return 0;
}



error_code WriteDAC( unsigned short *pchan, unsigned char InUse,
                    int IsTPC)
{
  int n;
  unsigned char k=1;

  for( n=0; n<NDAC; n++) {
    if( InUse & k) FakeDAC[n]= *pchan;
    pchan++;
    k <<= 1;
  }
  return 0;
}



error_code ReadRegister( unsigned short *preg, int IsIn)
{
//  FakeInReg= 0xCAFE, FakeOutReg= 0xFADE;
  if( IsIn) *preg= FakeInReg;
  else      *preg= FakeOutReg;
}


error_code WriteRegister( unsigned short reg)
{
  FakeOutReg= reg;
}

#endif

