/*
   msgsvrin : Local Station Software - Pierre Auger Project
   **********
                             ----------------
              JM Brunet, L Gugliemi, C Le Gall, G Tristram
	               College de France - Paris

   Created    01/06/97 - JMB & CL
   Modified : 12/09/97 - Split MsgSvr into MsgSvrIn & MsgSvrOut
                       - Take care of the format of the communication Buffer
              16/11/98 - Version 2
	      16/03/99 - Version 2.2
	      18/05/99 - Version 2.2 : Add Frigo
	      22/02/00 - Version 2.3 : Add communication with Suscriber Unit
                                       Change in Frame & Header formats
	      18/10/00 - Version 2.6 : RS-232 Com. Protocol V2.0
  
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <alarm.h>
#include <types.h>
#include <string.h>
#include <sg_codes.h>

#define MAIN
#define MAX_NB_ADDRESS  5
#define MAX_PIECE      16    /* Se plante en Download pour 32 ou 64 !!! */
#define MAX_FRIGO      10
#define SUBYTES         5    /* Suscriber Unit Packet length, type 
			      StationID & Packet number */

#include "crc32.h"
#include "msgdefs.h"
#include "ready_acq.h"
#include "augererr.h"
#define _STATUS_MAIN_
#include "status.h"
#define _CONFIG_MAIN_
#define _MONITOR_MAIN_
#include "run_config.h"
#include "svrdm.h"

typedef struct {
  MsgTypeIn type;
  MSGPROCS  proc;
} TYPE_PROC;
                 /* Type & task name for all messages but those for CONTROL */
TYPE_PROC TypeProc[]= { 
  M_T3_YES,    EVTSVR,
  M_MONIT_REQ, CALMON,
  M_CALIB_REQ, CALMON,
  M_GPS,       GPS,
};


signal_code Signal= 0;
FILE *FromSU= NULL;
char StrLog[256];
short NbAdd, TabAdd[MAX_NB_ADDRESS];
unsigned char NbSaveBuf= 0, SaveBufLength[MAX_PIECE+1];
unsigned char SaveBuf[MAX_PIECE+1][BUFCOMSIZE];
unsigned char NbFrigo= 0, FrigoLength[MAX_FRIGO];
unsigned char Frigo[MAX_FRIGO][BUFCOMSIZE];
unsigned char FrigoNumber[MAX_FRIGO];
CONFIG *cf;
STATUS  *st;

void  init( void);
int   ReadRS232( void);
int   ReadChar( unsigned char *p, int n);
int   CheckBufIn( void);
short DecodeAddress( void);
void  ResetBuffer( unsigned char *pc, unsigned char nb);
void  SendToMbx( unsigned char *pheader);
void dmp( char *title, unsigned char *p, int size);
short PrtAddress( void);
error_code set_read_tmout( path_id path, u_int32 tmout);
error_code reset_popt( path_id path );


byebye( error_code err)
{
  int n;
  
  sprintf( StrLog, "msgsvrin: ended with error %d\n", err);
  PrtLog( StrLog, 1);
                             /* Close all Mailbox */
  for( n=0; n<PROCESS_NB; n++) if( Process[n].is_in) {
    if(( err= PostClose( &Process[n].mbx_in)) != E_SUCCESS)
      printf( "msgsvrin : PostClose failed : error= %d\n", err);
  }
  printf( "msgsvrin : all mailbox are closed ... Ouf !!!!\n");
  fclose( FromSU);
  dm_finish( &SvrHeader );
  exit( err);
}


void sig_hand( signal_code s )
{
  int n;
  Signal= s;
  if( Signal == SIGQUIT) byebye( 0);
  _os_rte();
}

main( int argc, char **argv )
{
  unsigned char *pcom, *pc, *pr;
  unsigned char completion, slice, type, version, nbytes, npiece;
  unsigned char MaxLength= BUFCOMSIZE- sizeof( MESSAGE_HEADER)- 4;
  unsigned long *plong;
  u_int32 nt;
  int err, ok, IsFirstFrame= 1, ntt;
  signal_code dummy;
  short skip, IsComplete;
  MESSAGE_HEADER header;

  init();
  while( 1) {
  }
}



void init( void)
{
  error_code err;
  int n;
  unsigned int ready_mask;
  PROCESS *pproc;
  signal_code sig;
  u_int32 tmout= 10;                   /* Timeout on read */
  u_int32 nt, ndata;

                                       /* Link to Data Modules */
  if( dm_init( SvrModName, sizeof( SVRDM), (void **)&SvrData, &SvrHeader))
    exit(_errmsg(errno, "msgsvrin : Can't create SVRDM data module!\n"));
  if( dm_init( StatModName, sizeof( STATUS), (void **)&StData, &StHeader))
    exit(_errmsg(errno, "msgsvrin : Can't create STATUS data module!\n"));
  if( dm_init( ConfigModName, sizeof( CONFIG), (void **)&CfData, &CfHeader))
    exit(_errmsg(errno, "msgsvrin : Can't create CONFIG data module!\n"));
  st= StData;
  cf= CfData;
  st->ComIn.nb= 0;
  if(( err= ReadyInit( &ReadyEvId, ReadyEvName))) {
    sprintf( StrLog, "msgsvrin : Error %d calling ReadyInit\n", err);
    PrtLog( StrLog, 1);
    byebye( err);
  }
  if(( err= ReadyToGo( ReadyEvId, READY_MSGSVRIN))) {
    sprintf( StrLog, "msgsvrin: Call ReadyToGo(): err= %d\n", err);
    PrtLog( StrLog, 1);
    byebye( err);
  }
  ready_mask= READY_CONTROL | READY_TRIGGER2 | READY_EVTSVR;
  if(( err= ReadyAll( ReadyEvId, &sig, ready_mask)) != E_SUCCESS) {
    sprintf( StrLog, "msgsvrin : Unexpect sig %d call ReadyAll() : err= %d\n",
	     sig, err);
    PrtLog( StrLog, 1);
    byebye( err);
  }
                                       /* Open Mailbox in send mode */
  for( n=0; n<PROCESS_NB; n++) if( Process[n].is_in) {
    pproc= Process+n;
    if(( err = PostLink( &pproc->mbx_in, pproc->pnamein ))) {
      sprintf( StrLog, 
	    "msgsvrin : Can't Open Mailbox %s ($%X) for Send : err= %d\n",
			   pproc->pnamein , pproc->mbx_in, err);
      PrtLog( StrLog, 0);
    }
    else printf( "msgsvrin : Mailbox %s ($%X) opened for Send\n",
	      pproc->pnamein , pproc->mbx_in);
  }

  if(( err= ReadyToGo( ReadyEvId, READY_MSGSVRIN))) {
    sprintf( StrLog, "msgsvrin: Call ReadyToGo(): err=%d\n", err);
    PrtLog( StrLog, 1);
    byebye( err);
  }
                                       /* Open a pipe on RS232 read line */
  printf( "msgsvrin : Try to open the RS232 pipe\n");
  while((FromSU= fopen( RS232FromCentral, "r")) == NULL)
    {
      /*      printf("msgsvrin: problem in opening RS232 pipe");
	      fflush(stdout);*/
      nt= 1; _os_sleep( &nt, &sig);      /* Wait for Central to open it */
    }
  printf( "msgsvrin: pipe %s opened in read mode\n", RS232FromCentral);
  ResetBuffer( SaveBufLength, MAX_PIECE);
  ResetBuffer( FrigoLength, MAX_FRIGO);
                                       /* Intercept Signals and mask them */
  _os_intercept( sig_hand, _glob_data );
//  set_read_tmout( 0, tmout);           /* Set timeout on read */
  err= ReadyFinish( ReadyEvId, ReadyEvName);
  printf( "msgsvrin : init ended...\n"); fflush( stdout);
}


void ResetBuffer( unsigned char *pc, unsigned char nb)
{
  while( nb--) *pc++= 0;
}


void SendToMbx( unsigned char *pheader)
{
  unsigned char *pmess, *p1, *p2;
  int nbytes, n, m;
  TYPE_PROC *tp;
  PROCESS *pproc;
  MSGPROCS proc;
  error_code err;
  u_int32 nt;
  signal_code dummy;
  MESSAGE_HEADER *phead= (MESSAGE_HEADER *) pheader;
                        /* In which MailBx must I write the data ?
			   If I don't know, I send it to control */
  tp= TypeProc; proc=CONTROL;
  while( tp->type != M_GPS) {
    if( phead->type == tp->type) { proc= tp->proc; break; }
    tp++;
  }
  if(( pproc= myprocess( proc)) == NULL) {
    PrtLog( "msgsvrin: error calling myprocess()\n", 1); 
    byebye( 2001);
  }
                  /*** Allocate a buffer for the corresponding MailBx */
  nbytes= sizeof( MESSAGE_HEADER)+ phead->length+10;
  while(( err= PostAlloc( pproc->mbx_in, (void **)&pmess,
			  nbytes)) != E_SUCCESS) {
    if( err == POST_ERR_FULL) {
      nt= 1; _os_sleep( &nt, &dummy);     /* Mailbox full, try again */
      continue;
    }
    sprintf( StrLog, "msgsvrin: PostAlloc: err= %d\n", err);
    PrtLog( StrLog, 1);
    byebye( err);
  }
                        /*** And write data to the MailBox */
  p1= pmess; p2= pheader;
  for( n=0; n< sizeof( MESSAGE_HEADER); n++) *p1++= *p2++;
  for( n=0; n< NbSaveBuf; n++) {
    p2= SaveBuf[n];
    for( m=0; m<SaveBufLength[n]; m++) *p1++= *p2++;
  }
                        /*** Then, send the message      */
  if( (err= PostSend( pproc->mbx_in, (void *)pmess)) != E_SUCCESS) {
    sprintf( StrLog, "msgsvrin: PostSend: err= %d\n", err); 
    PrtLog( StrLog, 1);
  }
}


int ReadRS232( void)
{
  static unsigned char preamb[]= "!SU2LS!";
  int ndata=1;
  unsigned char  npreamb=0, *p=preamb;
  static unsigned char temp[COMBUFMAX];

  while( 1) {                                 /* Check Preamble */
    ndata= 1;
    //ReadChar( temp, ndata);
    if( !npreamb & temp[0] == ' ') continue;
    if(temp[0] != *p++) {
      sprintf( StrLog, "msgsvrin: bad preamb: npreamb= %d - char= $%X\n", 
	       npreamb, temp[0]);
      PrtLog( StrLog, 1);
      return 1;
    }
    if( ++npreamb == 7) break;
    else continue;
  }
  ndata= 1;
  ReadChar( BufCom, ndata);
  if((ndata= BufCom[0]) > BUFCOMSIZE+ 6) {     /* Bad length */
    sprintf( StrLog, "msgsvrin: Bad Length from SU: %d\n", ndata);
    PrtLog( StrLog, 1);
    return 1;
  }
  ReadChar( &BufCom[1], ndata);
  switch( BufCom[1]) {
  case SU_DATA_ACK :             /**** Data Acknowledge */
    SvrData->IsSent= 0; return 3;
  }
  return 0;
}

int ReadChar( unsigned char *p, int n)
{
  signal_code dummy;
  u_int32 nt;
  int ndata=n;
  error_code err;
  static int ndataold=0;
 
  while( 1) {
    //_os_gs_ready( fileno(FromSU), (void *)&ndata);
    if( ndata >= n) break;
    /************* Correction a etudier (SU bloque)
    if( n>1 && ndata && ndataold == ndata) { 
      sprintf( StrLog, "@!@!@!@!@!@!@!@!@ : %d\n", ndata);
      PrtLog( StrLog, 1);
      return 1;
    }
    else ndataold= ndata;
    if( n == 1) ndataold= 0;
    *************/
    nt= (n-ndata)/27;              /* 38400 bauds (/8 for 9600 bauds) */
    if( nt < 2) nt= 2;
    _os_sleep( &nt, &dummy);
  }
  err= _os_read( fileno( FromSU), p, (void *)&n);
  if( err) {
    sprintf( StrLog, "msgsvrin: _os_read err= %d\n",  err);
    PrtLog( StrLog, 1);
  }
  return err;
}


int CheckBufIn( void)  /* Check buffer from SU :
			    - Check packet integrity (CRC & ETX)
			    - Check type
			  Return values:
			    - 0 : data from Central
			    - 1 : need acknowledge from msgsvrout
			    - 2 : bad packet integrity
			    - 3 : nothing more to do for this packet
		       */
{
  unsigned char *p, *pcom= BufCom ;
  unsigned long crc, crcp;
  int ndata= BufCom[0], type= BufCom[1];
  int nb;

  crc= crc32( BufCom, ndata-4);   /* Check CRC */
  if( memcmp( &crc, BufCom+ ndata-4, 4)) {
    memcpy( &crcp, BufCom+ ndata-4, 4);
    sprintf( StrLog, "msgsvrin: Bad CRC: $%X - $%X \n", crcp,crc); 
    PrtLog( StrLog, 1);
    return 2;
  }
  if( *(BufCom+ ndata) != 0xFF) { /* Check ETX */
    sprintf( StrLog, "msgsvrin: Bad ETX: %d\n",*(BufCom+ ndata));
    PrtLog( StrLog, 1);
    return 2;
  }
  type= *(BufCom+1);              /* Action from Type value */
  switch( type) {
  case SU_DATA:                   /**** Data */
    SvrData->IsDataAck= 1;
    SvrData->BSid= *(unsigned short *)(BufCom+2);
    SvrData->PacketNumberRec= *(BufCom+4);
    return 0;
  case SU_DATA_ACK:               /**** Data Acknowledge */
    SvrData->IsSent= 0; return 3;
  case SU_ECHO:                   /**** Echo */
    memcpy( SvrData->EchoData, BufCom+2, ndata-6);
    SvrData->IsEchoReq= ndata-5; break;
  case SU_ECHO_ACK:               /**** Echo Acknowledge */
    return 3;
  case SU_CONTROL:                /**** Control */
    SvrData->IsControlCommand= 1;
    break;
  case SU_CONTROL_ACK:            /*** Control Acknowledge */
    return 3;
  case SU_POWER:                  /**** Power Fail (What to do ??????) */
    SvrData->IsFail= 1; return 3;
  case SU_RESET:                  /**** Software reset      */
    PrtLog( "msgsvrin : Software Reset Received from SU\n", 0);
    SvrData->IsSoftReset= 1; break;
  case SU_RESET_ACK:              /**** Software reset acknowkedge */
    return 3;
  case SU_STATUS:                 /**** Reset + Status Broadcast */
    return 3;
  case SU_VERSION:                /**** Version Request */
    SvrData->IsVersionReq =1; break;
  case SU_VERSION_ACK:            /**** Version reply */
    return 3;
  case SU_GPS:                    /**** GPS 1PPS Status Request */
    SvrData->IsGpsStatusReq= 1; break;
  case SU_ID:                     /**** Station ID Request */
    SvrData->IsStationIdReq=1; break;
  case SU_NETWORK_REPLY:          /**** Network Status */
    SvrData->NetworkStatus= *(BufCom+2);
    if( SvrData->NetworkStatus != 0x02) 
      SvrData->IsSent= 0;
    return 3;
  case SU_INVALID:                /**** Command Invalid */
    SvrData->IsCommandInvalid =1; break;
  default:              /**** ask msgsvrout to send a 'Command Invalid' */
    sprintf( StrLog, "msgsvrin: Unkwown type ($%X) from SU\n", type);
    PrtLog( StrLog, 1);
    SvrData->SendCommandInvalid =1; fflush( stdout); break;
  }
  return 1;
}


short DecodeAddress( void)  /* return the number of bytes to skip in BufCom */
{                           /* or zero if the station was not addressed     */
                            /* The station appears NbAdd times in the list  */
               /* TabAdd[] contains the position of the Station in the list */
  unsigned char *pcom= BufCom+SUBYTES;
  int DestinationFlag;
  short add, nbadd, n;
  unsigned short hard= cf->AcqParams.HardAddress;
  /*  unsigned short logi= cf->AcqParams.LogicalAddress;*/ /* BR-FC */

  pcom += 4;                     /* skip length & reserved bytes */
  NbAdd= 1;
  DestinationFlag= (*pcom & 0xC0) >> 6;
  switch( DestinationFlag) {
  case 0:                        /* Single address */
    add= *pcom++ << 8;
    add |= *pcom++; /* on concatene les 2 bytes dans le short add (1 short=2 bytes)*/
   if( add == hard)         return 6+SUBYTES;
    /* if( logi && add == logi) return 6+SUBYTES;*/ /* BR-FC */
   else return 0;
  case 3:                        /* All stations broadcast */
    return 6+SUBYTES;
  case 1:                        /* List of stations */
    NbAdd= 0;
    pcom++;
    nbadd= *pcom++;
    for( n=0; n<nbadd; n++) {
      add= *pcom++ << 8; add |= *pcom++;
     /*      if( add == hard || (logi && add == logi)) TabAdd[NbAdd++]= n;*/ /*BR-FC */
      if( add == hard ) TabAdd[NbAdd++]= n;
    }
   if( !NbAdd) return 0;
    return ((nbadd+1) << 1)+4+SUBYTES;
  case 2:                        /* Anti List of stations */
    pcom++;
    nbadd= *pcom++;
    for( n=0; n<nbadd; n++) {
      add= *pcom++ << 8; add |= *pcom++;
      /* if( add == hard || (logi && add == logi)) return 0;*/ /*BR-FC*/
      if( add == hard ) return 0;
    }
    return ((nbadd+1) << 1)+4+SUBYTES;
  }
}



void dmp( char *title, unsigned char *p, int size)
{
  int n;

  printf( "msgsvrin: %s : size= %d", title, size); fflush( stdout);
  for( n=0; n<size; n++) {
    if( !(n % 20)) printf( "\n");
    printf( "%3X", *p++);
  }
  printf( "\n");
  fflush( stdout);
}    



short PrtAddress( void)     /* return the number of bytes to skip in BufCom */
{
  unsigned char *pcom= BufCom+SUBYTES;
  int DestinationFlag;
  int add;
  short nbadd, n;

  DestinationFlag= (*pcom >> 6);
  printf( "Station Address : $%X (hard) - $%X (Logical)\n",
	  cf->AcqParams.HardAddress, cf->AcqParams.LogicalAddress);
  switch( DestinationFlag) {
  case 0:                        /* Single address */
    add=  ((*pcom++ & 0x3F) << 8) | *pcom;
    printf( "Single Address : $%X\n", add);
    return 2;
  case 3:                        /* All stations broadcast */
    printf( "All stations broadcast\n");
    return 2;
  case 1: case 2:                /* List or Anti-List of stations */
    if( DestinationFlag == 1) printf( "List of stations :\n");
    else printf( "Anti-List of stations :\n");
    pcom++; nbadd= *pcom++;
    for( n=0; n<nbadd; n++) {
      if( !(n % 8)) printf( "\n  ");
      add= ((*pcom++ & 0x3F) << 8) | *pcom;
      printf( "%9X", add);
    }
    return( (nbadd+1) << 1);
  }
}



/****************************************************************/

