/*
   msgsvrout : Local Station Software - Pierre Auger Project
   ---------
                             ----------------
              JM Brunet, L Gugliemi, C Le Gall, G Tristram
	               College de France - Paris

   Created  : 01/06/97 - JMB & CL
   Modified : 12/09/97 - Split MsgSvr into MsgSvrIn & MsgSvrOut
                       - Take care of the format of the communication Buffer
              17/11/98 - Version 2
	      20/11/98 - Change StoreData philosophy : allow multiple
	                 fields for message of a same type.
	      19/03/99 - Version 2.2 : Changes in MESSAGE_HEADER
	      25/02/00 - Version 2.3 : Add communication with Suscriber Unit
                                       Change in Frame & Header formats
	      18/10/00 - Version 2.6 : Coms Protocol V2.0
	      30/10/00 - Replace Cycle Alarm by Event on GPS 1 PPS
----------------------------------------------------------------------------
   Purpose :   Loop over all output mailboxes and send messages to the Central 
   ---------   Station via the Communication Buffer.
----------------------------------------------------------------------------*/


#include <stdio.h>
#include <string.h>
#include <process.h>
#include <errno.h>
#include <cglob.h>

#define MAIN
#define BUFSIZE  BUFSIZEX     /* Must fit PROCESS definition (see msgdefs.h) */
#define PREAMBLE 9
#define SUBYTES  PREAMBLE+5   /* preamble+ length+ type*/ 
				/* + Station ID + packet number*/
#define SUERRORS 10           /* Reset SU if too many errors */

#define POWERON "POWERON"
#define FAKE_GPS
#define FAKE_SU
			/* to simulate the 1pps by soft */
#define SIG_ALARM  14       /* Signal for Cycle Alarm */

#include "crc32.h"
#include "msgdefs.h"
#include "ready_acq.h"
#include "augererr.h"
#define _STATUS_MAIN_
#include "status.h"
#define _CONFIG_MAIN_
#define _MONITOR_MAIN_
#include "run_config.h"
#include "svrdm.h"

extern unsigned int HARD_ADDRESS;
CONFIG *cf;
STATUS  *st;

signal_code Signal= 0;
process_id IrqGpsPid;      /* extirq pid for GPS 1 pps interrupt */
event_id Gps1ppsId;        /* Event Id   for GPS 1 pps interrupt */
int NbData= 0;
char StrLog[256];
FILE *ToSU;
MESSAGE *pM[PROCESS_NB+1];
extern int errno;  

void init( void);
void DoWork( void);
void DumpRS232( int flag);
unsigned char StoreData( unsigned char *pcom, PROCESS *pproc, int *nbfields);
void dmp( char *title, unsigned char *p, int size);
void reset( void);
void byebye( error_code err);
int  SendPacket( int cmd);
int  Verify( void);
void WrtRS232( void);
unsigned char GpsStatus( void);

void byebye( error_code err)
{
  int n;
  
  sprintf( StrLog, "msgsvrout: ended with error %d\n", err);
  PrtLog( StrLog, 1);
                             /* Close all Mailbox */
  for( n=0; n<PROCESS_NB; n++) if( Process[n].is_out) {
    if(( err= PostClose( &Process[n].mbx_out)) != E_SUCCESS)
      printf( "msgsvrout : PostClose failed : error= %d\n", err);
  }
  printf( "msgsvrout : all mailbox are closed ... Ouf !!!!\n");
  fclose( ToSU);
  dm_finish( &CfHeader);
  dm_finish( &StHeader);
  dm_finish( &SvrHeader);
  exit( err);
}


void sig_hand( signal_code s )
{
  int n;
  Signal= s;
  _os_rte();
}

main( int argc, char **argv )
{
  error_code err;
  signal_code sig;
  unsigned int value;

  init();
  strcpy((char *)BufCom,"  !LS2SU!");    /* Write preamble, 9 bytes */
  SendPacket( SU_STATUS);                /* Send Reset + Status broadcast  */
  /*  dmp("*** BufCom",BufCom,PACKETSIZE);*/

#ifdef FAKE_SU
  while( 1)
    {                            /***** Start infinite Loop *****/
      Signal= 0;
      value=0;
      /* Wait for something to hapened */
      err= _os_sleep( &value, &sig);
      if( Signal == SIGQUIT) byebye( 2000);
      Verify();
      if( Signal != SIG_SVR) DoWork();
    }
#else
  while( 1)
    {                            /***** Start infinite Loop *****/
      Signal= 0;
      /* Wait for something to hapened */
      err= _os_ev_wait( Gps1ppsId, &value, &sig, 1, 0xfffffff);
      if( Signal == SIGQUIT) byebye( 2000);
      if( value == 1 && st->RunStatus == REBOOT) break;
      Verify();
      if( Signal != SIG_SVR) DoWork();
    }                                        /* End infinite loop */
#endif

  sprintf( StrLog, "msgsvrout: Ended by signal %d\n", Signal);
  PrtLog( StrLog, 1);
  byebye( 0);
}


void DoWork( void)
{
  int n, nbfields=0;
  PROCESS *pproc;
  error_code err;
  unsigned char *pnbfield= BufCom+6+SUBYTES, *pcom;
  unsigned char nwrite;
  unsigned char nbytes;
  static unsigned char frame_nb= 0;
                                    /* Write Source */
  pcom= BufCom+SUBYTES;
  pcom += 4;                        /* reserve space for the 4 first bytes */
  *pcom++= cf->AcqParams.HardAddress >>8;
  *pcom++= cf->AcqParams.HardAddress & 0xFF;

  *pnbfield= 0;                     /* Reset Nb of messages field   */
  nbytes= 7;                        /* Frame length                 */
  pcom= pnbfield+1;                 /* Start first message add.     */
                                    /*--- Loop over all MailBox     */
  for( n=0; n<PROCESS_NB; n++) if( Process[n].is_out) {
    pproc= Process+n;
    nwrite= StoreData( pcom, pproc, &nbfields);
    if( !nwrite) continue;
    nbytes += nwrite;
    pcom += nwrite+ nbfields;       /* Nb of bytes+ message        */
    if( !nbfields) nbfields++;
    *pnbfield += nbfields;          /* increment nb of fields      */
    if( nbytes > BUFCOMSIZE+5) break;
  }                                 /*--- End loop over MailBox ---*/
                                    /* If there is something to    */
  if( *pnbfield) {		    /* write, send packet to SU    */
    NbData= nbytes+*pnbfield;       /* Frame user length           */
    BufCom[SUBYTES+1]= frame_nb++;  /* #Frame                      */
    st->ComOut.nb++;
    st->ComOut.cursize  = NbData;
    st->ComOut.totsize += NbData;
    SendPacket( SU_DATA);         /* Send Packet : data          */
  }
}



void init( void)
{
  error_code err;
  int n, lsize;
  unsigned int ready_mask;
  PROCESS *pproc;
  signal_code sig;
  alarm_id AlarmId;
  char *argblk[]= { "extirq", "-n3", "-p5", "-sGps1pps", "-ep", 0};
  u_int16 prior=0, paths=3;  /* prioriy and nb of paths to inherit */
  u_int32 edata= 0;          /* No Extra Data Space */
  char orphan= 0;            /* make this child a normal child */
  u_int16 type_lang= mktypelang( MT_PROGRAM, ML_OBJECT);

  printf( "msgsvrout : MESSAGE size= %d\n", sizeof( MESSAGE)); fflush( stdout);
                             /* Fork external interupt for GPS 1 PPS */
#ifndef FAKE_GPS
  if( (err= _os_exec( _os_fork, prior, paths, argblk[0], argblk, _environ,
		     edata, &IrqGpsPid, type_lang, orphan)) != SUCCESS) {
    sprintf( StrLog, "msgsvrout : problem forking %s\n", argblk[0]);
    PrtLog( StrLog, 1);
    exit( _errmsg( errno, StrLog));
  }
  printf( "msgsvout : %s forked : Pid= %d\n", argblk[0],IrqGpsPid );
#endif

                             /* Link to Data Modules */
  if( dm_init( ConfigModName, sizeof( CONFIG), (void **)&CfData, &CfHeader))
    exit(_errmsg(errno, "msgsvrout : Can't link to Config data module!\n"));
  if( dm_init( StatModName, sizeof( STATUS), (void **)&StData, &StHeader))
    exit(_errmsg(errno, "msgsvrout : Can't link to Status data module!\n"));
  if( dm_init( SvrModName, sizeof( SVRDM), (void **)&SvrData, &SvrHeader))
    exit(_errmsg(errno, "control : Can't create SVRDM data module!\n"));
  st= StData;
  cf= CfData;
  st->ComOut.nb= 0;
  SvrData->Pid= getpid();
                             /* Open Mailbox in read mode */
  if(( err= ReadyInit( &ReadyEvId, ReadyEvName))) {
    sprintf( StrLog, "msgsvrout: ReadyInit: err= %d\n", err);
    PrtLog( StrLog, 1);
    byebye( err);
  }
  for( n=0; n<PROCESS_NB; n++) if( Process[n].is_out) {
    pproc= Process+n;
    if((err= PostCreate( &pproc->mbx_out, pproc->pnameout, pproc->Msize_out,
			 &NoNotify))) {
      sprintf( StrLog, "msgsvrout:Can't Create Mailbox %s: err= %d\n", 
	      pproc->pnameout, err ) ;
      PrtLog( StrLog, 1);
      byebye( err ) ;
    }
    else 
      printf( "msgsvrout : Mailbox %s (%d) created with size= %d\n", 
	      pproc->pnameout, pproc->mbx_out, pproc->Msize_out);
    fflush( stdout);
  }
                             /* Open on RS232 Communication line */
  if( (ToSU= fopen( RS232ToCentral, "w")) == NULL) {
    sprintf( StrLog, "msgsvrout: Cant't open pipe %s for write\n",
			 RS232ToCentral);
    PrtLog( StrLog, 1);
    byebye( errno);
  }
  printf( "msgsvrout : pipe %s opened for write\n",  RS232ToCentral);
  fflush( stdout);
                     /* The other processes can now link to all the MailBox */
  
  if(( err= ReadyToGo( ReadyEvId, READY_MSGSVROUT))) {
    sprintf( StrLog, "msgsvrout: ReadyToGo(): err= %d\n", err);
    PrtLog( StrLog, 1);
    byebye( err);
  }
               /* Allocate a buffer per process to save the mailbox mess. */
  printf( "msgsvrout : allocate mes. buffers (process #%d)\n", pproc->procs);
  fflush( stdout);
  for( n=0; n<PROCESS_NB; n++) { 
    pproc= Process+n;
    lsize= pproc->Bsize_out+ sizeof( MESSAGE_HEADER)+ 10;
    pM[n]= (MESSAGE *)malloc( lsize);
    if( pM[n] == NULL) {
      sprintf( StrLog, "msgsvrout: can't alloc mes. buffer #%d\n", n);
      PrtLog( StrLog, 1);
    }
    else printf( "msgsvrout : mes. buffer #%d allocated - size %d\n",
		 n, lsize);
    fflush( stdout);
  }

#ifdef FAKE_GPS			/* simulate 1pps with a soft GPS */
                                    /* Set a Cycle Alarm */
  if(( err= _os_alarm_cycle( &AlarmId, SIG_ALARM, 0x80000100)))
  {
    sprintf( StrLog, "msgsvrout: _os_alarm_cycle(): err= %d\n", err);
    PrtLog( StrLog, 1);
  }

#else					/* use the actual GPS */
                        /* Link to Gps1pps event */
  _os_send( IrqGpsPid, 1000);         /* Enable Gps 1 Pps IT */
  if(( err= _os_ev_link( "Gps1pps", &Gps1ppsId)) != SUCCESS)
  {
    strcpy( StrLog, "msgsvrout : Can't link to Gps1pps event\n");
    PrtLog( StrLog, 1);
    exit( _errmsg( errno, StrLog));
  }
#endif						/* modified by BR 5/12/00 */
                                      /* Intercept Signals and mask them */
  _os_intercept( sig_hand, _glob_data );
  err= ReadyFinish( ReadyEvId, ReadyEvName);
  printf( "msgsvrout : init ended...\n"); fflush( stdout);
}


unsigned char StoreData( unsigned char *pcom, PROCESS *pproc, int *nbfields)
{
  unsigned char *pnbytes, *pcompletion, *ptype, *pversion, *pdata, *p1, *p2;
  unsigned char nwrite=0, nbytes, slice;
  int to_read= 0, nfree;
  int lsize, n;
  unsigned char previous, is_stop=0;
  char line[80];
  error_code ok;

  *nbfields= 0;
  while( !is_stop) {
    pnbytes= pcom++;           /* pointer on nb. of bytes for this field */
    *pnbytes= 0;
    pcompletion= pcom++;       /* pointer on Completion + position */
    ptype= pcom++;             /* pointer on Type */
    pversion= pcom++;          /* pointer on Version */
    nwrite += 3;
    nbytes= 3;
    to_read= pproc->prev_read;
    nfree= BUFCOMSIZE- (pcom - BufCom) +5;  /* Compute space left */
    if( nfree < 0 || nfree > BUFCOMSIZE- 8) {
      return nwrite -3;
    }
    if( to_read) {             /* Continue previous mailbox...    */
      previous= 1;
      *ptype=   pproc->prev_type;
      *pversion= pproc->prev_version;
      pdata=    pproc->prev_pdata;
    }
    else {
      previous= 0;            /* if nothing in Mailbox : skip it */
      if(( ok= PostReceive( pproc->mbx_out, (void **)&pMessage, &lsize))
	 != E_SUCCESS) return nwrite -3;
                  /*      read first int, put it into 'to_read' variable
		          read next char, put it into *ptype 
			  'to_read'= nb of bytes of data    */
      lsize -= sizeof( MESSAGE_HEADER);
      if( lsize <0 ) lsize= 0;
      if( lsize > (pproc->Bsize_out+ sizeof( MESSAGE_HEADER))) {
	sprintf( StrLog, "msgsvrout: proc #%d: mess size= %d: too big\n",
		pproc->procs, lsize);
	PrtLog( StrLog, 1);
	PostCancel( pproc->mbx_out, pMessage);
	return nwrite -4;
      }
      pM[pproc->procs]->mheader.length= pMessage->mheader.length;
      pM[pproc->procs]->mheader.type= pMessage->mheader.type;
      pM[pproc->procs]->mheader.version= pMessage->mheader.version;
      p1= pM[pproc->procs]->buf;
      p2= pMessage->buf;
      while( lsize--) *p1++= *p2++;
      PostCancel( pproc->mbx_out, pMessage);
      to_read=   pM[pproc->procs]->mheader.length-sizeof(MESSAGE_HEADER);
      *ptype=    pM[pproc->procs]->mheader.type;
      *pversion=  pM[pproc->procs]->mheader.version;
      pdata=     pM[pproc->procs]->buf;
    }
    if( previous)               /* Define message slice          */
      pproc->prev_slice += 1;
    else {
      pproc->prev_slice= 0;
    }
    *pcompletion= slice= pproc->prev_slice;
    if( to_read > nfree) {      /* Define to_read and set TypeFlag */
      pproc->prev_read=    to_read- nfree;
      pproc->prev_slice=   slice;
      pproc->prev_type=    *ptype;
      pproc->prev_version= *pversion;
      if( previous) *pcompletion |= COMPLETION_NEXT;
      else {
	*pcompletion |= COMPLETION_FIRST;
	pproc->messnb += 1;
      }
      pproc->prev_pdata= pdata+ nfree;
      to_read= nfree;
      is_stop= 1;
    }
    else {
      if( previous) { *pcompletion |= COMPLETION_LAST; pproc->prev_read= 0; }
      else            *pcompletion |= COMPLETION_ALL;
    }
    *pversion    |= (pproc->messnb << 2);
                             /* read 'to_read' bytes from the Mailbox
			        and put them into *pcom */
    for( n=0; n<to_read; n++, nbytes++) *pcom++= *pdata++;
    *nbfields += 1;
    nwrite += to_read;
    *pnbytes= nbytes;
  }
  return nwrite;
}


int Verify( void)
{
  unsigned char *plength= BufCom+PREAMBLE, *ptype= plength+1, *pdata= ptype+1;
  static int Error=0;

  if( SvrData->IsSent) {             /* No acknowledge : send packet again */
    PrtLog( "msgsvrout: No Acknow.: send packet again\n", 1);
    SendPacket( SU_DATA);
dmp( "Resent buffer", BufCom, *(BufCom+PREAMBLE)+10);
    Error++;
  }
  if( SvrData->SendCommandInvalid) { /* msgsvrin received an unknown type */
    PrtLog( "msgsvrout: Unknow type rec by msgsvrin\n", 1);
    SvrData->SendCommandInvalid= 0;
    SendPacket( SU_INVALID); Error++;
  }
  if( SvrData->IsCommandInvalid) {   /* Invalid command from SU : send again */
    PrtLog( "msgsvrout : Invalid Command from SU\n", 1);
    SvrData->IsCommandInvalid= 0;
    WrtRS232(); Error++;
  }
  if( Error > SUERRORS) {                 /* Too many errors  : Reset SU*/
    SendPacket( SU_RESET); Error=0;
    SvrData->IsSent= 0;
    SvrData->SendCommandInvalid= 0;
    SvrData->IsCommandInvalid=0;
  }
  if( SvrData->IsEchoReq) {          /* Echo Acknowledge   */
    printf( "msgsvrout : Echo Ack\n");
    SvrData->IsEchoReq= 0;
    SendPacket( SU_ECHO_ACK);
  }
  if( SvrData->IsSoftReset) {        /* Soft Reset requested */
    SvrData->IsSoftReset= 0;
    SendPacket( SU_RESET_ACK);
    sleep( 1);
    system( "reboot&");
  }
  if( SvrData->IsVersionReq) {       /* Version requested */
    SvrData->IsVersionReq= 0;
    SendPacket( SU_VERSION_ACK);
  }
  if( SvrData->IsDataAck) {          /* Data Acknowlege */
    SvrData->IsDataAck= 0; Error=0;
    SendPacket( SU_DATA_ACK);
  }
  if( SvrData->IsControlCommand) {   /* Control Command Acknowledge */
    SvrData->IsControlCommand= 0;
    SendPacket( SU_CONTROL_ACK);
  }
  if( SvrData->IsGpsStatusReq) {     /* GPS 1PPS Status Request */
    SendPacket( SU_GPS);
    SvrData->IsGpsStatusReq= 0;
  }
  if( SvrData->IsStationIdReq) {       /* Station ID Request */
    SendPacket( SU_ID_REPLY);
    SvrData->IsStationIdReq= 0;
  }
  fflush( stdout);
}


int SendPacket( int cmd)  /* No control command should be issued under normal
			     operating conditions. So I skip them.... */
{
  unsigned char *plength= BufCom+PREAMBLE, 
    *ptype= plength+1, *pdata= ptype+1;
  int ndata= NbData;
  unsigned char *p;
  static unsigned char echostring[]= "LocalStationEcho";
  static unsigned char versionstr[]= "V02.00";
  char *POWERON_value=NULL;

  *ptype= cmd;
  switch( cmd) {
  case SU_DATA:                     /* Data Command */
    memcpy( pdata, &SvrData->LSid, 2);
    pdata += 2;
    *pdata++ = SvrData->PacketNumberSent;
    SvrData->PacketNumberSent += 1;
    ndata += 3;                     /* add StationID & packet number */
    *plength= 6+ndata; 
    *pdata= *plength-10;            /* Frame length */
     SvrData->IsSent= 1;
     /* dmp( "", pdata+1, ndata); */
    break;
  case SU_DATA_ACK:                 /* Data Acknowledge */
    memcpy( pdata, &SvrData->BSid, 2);
    pdata += 2;
    *pdata++ = SvrData->PacketNumberRec;
    *plength= 9; 
     SvrData->IsSent= 0;
    break;
  case SU_ECHO:                     /* Echo requested */
    ndata= 16;
    *plength= 6+ndata;
     p= echostring;
    while( ndata--) *pdata++= *p++;
    printf( "msgsvrout : Echo requested\n"); 
    break;
  case SU_ECHO_ACK:                 /* Echo Acknowledge */
    *plength= 6+ndata;
     p= (unsigned char *)SvrData->EchoData;
    while( ndata--) *pdata++= *p++;
    printf( "msgsvrout : Echo acknowledge\n"); 
    break;
  case SU_POWER:                         /* Power Fail */
    *plength= 6; 
     PrtLog( "msgsvrout: Power Fail\n", 0);
    break;
  case SU_RESET:                    /* Soft Reset Command */
    *plength= 6; 
    PrtLog( "msgsvrout : ***** RESET SU *****\n", 1); 
    break;
  case SU_RESET_ACK:                /* Soft Reset Acknowledge */
    *plength= 6;
     PrtLog( "msgsvrout: Soft Reset Ack & RESET\n", 0);
    break;
  case SU_STATUS:                  /* Reset & Status broadcast */
    /* a modifier */
    POWERON_value=getenv(POWERON);
    if( !POWERON_value )
      {
	printf("The %s environment variable doesn't exist\n",POWERON);
	exit(0);
      }
    else
      {
	if( !strcmp(POWERON_value,"Yes") ) *pdata=0x48;
	else if ( !strcmp(POWERON_value,"No") ) *pdata=0x53;
	printf("POWERON == %s, position in BufCom=%u\n",POWERON_value,pdata-BufCom);
	fflush(stdout);
      }
    *plength= 7;
    /*    *pdata= 0x48; */
    /* adresse mise en dur ; envoyer S ou H selon le type de reboot
       H = 0x48
       S = 0x53 */
    PrtLog( "msgsvrout: Reset & Status broadcast\n", 0);
    break;
  case SU_VERSION:                 /* Version request */
    *plength= 6;
     printf( "msgsvrout : Version requested\n"); 
    break;
  case SU_VERSION_ACK:             /* Version Reply */
     ndata= 6;
    *plength= 6+ndata;
     p= versionstr;
    while( ndata--) *pdata++= *p++;
    printf( "msgsvrout : Version reply\n"); 
    break;
  case SU_GPS:
    *plength= 7;
    *pdata= GpsStatus();
    break;
  case SU_ID_REPLY:
    *plength= 8;
    *(unsigned short *)pdata= SvrData->LSid;
    break;
  case SU_NETWORK:
    *plength= 6;
    break;
  case SU_INVALID:
    *plength= 6; 
    PrtLog( "msgsvrout: Invalide command\n", 1);
    break;
  default: sprintf( StrLog, "msgsvrout: unknown command ($%X)\n", cmd);
    PrtLog( StrLog, 1);
    return 1;
  }
  if( SvrData->NetworkStatus != 0x02) {
    sprintf( StrLog, "msgsvrout: Bad Network state : %d (mess cancelled)\n",
	     SvrData->NetworkStatus);
    PrtLog( StrLog, 1);
  }
  else WrtRS232();                       /* Send packet      */
  fflush( stdout);
}



void WrtRS232( void)           /* Send Communication Packet */
{
  unsigned long crc;
  int data_length= *(BufCom+PREAMBLE)-6;
  int lcrc= data_length+ 2;
  unsigned char *pcrc= BufCom+ data_length+11;
  u_int32 nb;
  
  crc= crc32( BufCom+9, lcrc);                 /* Compute CRC and store it */
  memcpy( pcrc, &crc, 4);
  *(pcrc+4)= 0xFF;                             /* ETX Field */
  nb= data_length+ 16;
  _os_write( fileno( ToSU), BufCom, &nb);      /* Send message */
}


unsigned char GpsStatus( void)   /* GPS status : return code :
				    0x00 : GPS 1PPS time mark invalid
				    0x01 : GPS 1PPS time mark valid
				 */
{
  if( StData->GpsState) return 1;
  return 0;
}

void dmp( char *title, unsigned char *p, int size)
{
  int n;

  printf( "msgsvrout: %s : size= %d", title, size); fflush( stdout);
  for( n=0; n<size; n++) {
    if( !(n % 20)) printf( "\n");
    printf( "%3X", *p++);
  }
  printf( "\n");
  fflush( stdout);
}    

void reset( void)
{
  unsigned char *pcom= BufCom;
  int n;

  for( n=0; n<BUFCOMSIZE; n++) *pcom++= 0xFF;
}

void DumpRS232( int flag)
                                /* Dump RS232 Buffer :
				   Flag= 0x01 : hexadecimal dump
				         0x02 : decimal dump
				         0x04 : decoded decimal dump */
{
  unsigned char *pcom= BufCom;
  unsigned char nfields, nbytes, type, completion, version, n, m, ndata;
  static int nevt= 1, source;

  nfields= *pcom++;
  printf( "******* Communication Record : %d (Send)\n", nevt++);
  if( flag & 0x1) {
    printf( "\n-------- Hexadecimal Dump ------------");
    for( m=0, pcom=BufCom; m<BUFCOMSIZE; m++) {
      if( !(m % 10)) printf( "\n%4X - ", m);
      printf( "%3X", *pcom++);
    }
    printf( "\n");
  }
  if( flag & 0x2) {
    printf( "\n-------- Decimal Dump ------------");
    for( m=0, pcom=BufCom; m<BUFCOMSIZE; m++) {
      if( !(m % 10)) printf( "\n%4d - ", m);
      printf( "%4d", *pcom++);
    }
    printf( "\n");
  }
  if( flag & 0x4) {
    printf( "\n-------- Decimal decoded Dump ------------\n");
    pcom= BufCom;
    source= (*pcom++ << 8) | *pcom++;
    printf( "Source= $%x --- ", source);
    nfields= *pcom++;
    printf( "Nb. of fields= %d\n", nfields);
    for( n=0; n<nfields; n++) {
      ndata= *pcom++-3;
      completion= *pcom++;
      type=  *pcom++;
      version= *pcom++;
      printf( "  Field %d - %d data - Type= %d - Version= %d - ", 
	      n, ndata, type, version);
      switch( completion & 0xC0) {
      case COMPLETION_ALL  : printf( "ALL");   break;
      case COMPLETION_FIRST: printf( "FIRST"); break;
      case COMPLETION_NEXT : printf( "NEXT (n= %d)", (completion & 0x3F));
	break;
      case COMPLETION_LAST : printf( "LAST (n= %d)", (completion & 0x3F)); 
	break;
      }	
      for( m=0; m<ndata; m++) {
	if( !(m % 12)) printf( "\n    ");
	printf( "%4d", *pcom++);
      }
      printf( "\n");
    }
  }
  fflush( stdout);
}
