/*

	tcpdatasvr.c
        AV, sep 2000 modified

	OVERVIEW:
	This program first redirects his standard input and output to file
	descriptor 4 (a BSD socket) inherited from the parent (tcpsvr).
	Then links to Data Modules and Buffers, to get access to the variables
	of the run.
	After that it enters a loop in which it reads a byte (command) from  
	stdin and sends the answer to stout, until the DISCONNECT command
	is received, or timeout arrives.

	COMMANDS:
	Valid commands are described in tcpsvr.h:

  		0 CONNECT	Request connection  
  		1 EVTBUF	Get Event Buffer
		2 MUONBUF 	Get Muon Buffer
		3 MONITBUF	Get Monitor Buffer
  		4 DSTATUS 	Get Status Data Module
		5 DCALIB	Get Calibration Data Module
  		6 DISCONNECT	Disconnect from server
  		7 NOP		Do nothing
  		8 LOOKSTAT	Get selected Status variables
		9 LOOKMON 	Get Monitor variables
	       10 LOOKMONPK	Get packed Monitor variables
	       11 CHANGEDA	...

	ANSWERS:
	They have the following general format:

	flag_byte:number_high_byte:number_low_byte:data_byte1:data_byte2...

	Valid answers are preceeded by a flag byte (options also described in
	tcpsvr.h)

  		0 OK		Command accepted
		1 NOK 		Unknown command
		2 TOUT		Time out occurred

	followed by the number of bytes that the answer has (may be 0) as
	an unsigned short (2 bytes), and at last, the answer as n bytes.

	TIMEOUT:
	A timeout is implemented to avoid hanging copies of this program, using
	os9 alarms. The default is set to 10s. If a command is not received in
	10s, it exits, sending TOUT message.
	The NOP command should be sent in interactive clients, when there is
	no command available, to avoid timeout.

	DEBUGGING:
	If debugging is enabled (Debug == 1), it writes messages to the file 
	tcpdatasvr.log in /
	

*/

#include<stdlib.h>
#include<const.h>
#include<errno.h>
#include<INET/in.h>

#include"tcpsvr.h"
#define _STATUS_MAIN_
#include"status.h"
#define _MONITOR_MAIN_
#define _CONFIG_MAIN_
#include "run_config.h"
#include "evbdefs.h"

#define TIME(secs) ((secs << 8) | 0x80000000)

int Debug = 1;

unsigned char command;
unsigned char flag = NOK;
unsigned short number = 0;
int nnnn;
unsigned int test = 0x01234567;
unsigned char *data;
unsigned char *datac;
unsigned short *datash;

/* A pointer to MONIT_PARAMS */
MONIT_PARAMS  *pconfig;

signal_code TimeOut = 0;

/* Short send (the header is allways the same)       */
void Send_Head( unsigned char , unsigned short);

/* New BLOCKING read & write                         */
/* (They don't return until they have ALL the bytes) */
int read_data(int , unsigned char *, int);
int write_data(int , unsigned char *, int);

/* PACK & UNPACK functions */
int pack(unsigned short *, int, unsigned char *);
int unpack(unsigned char *, int, unsigned short *);


/* Alarm Handler (the alarm is to prevent hanging childs) */
void alarm_hand(int s)
{
        /* The timeout alarm is here */ 
  TimeOut = 1;
  _os_rte();
}

int main(int argc, char **argv) {
  
  path_id path;
  int alarm_id;
  int secs = 10;
  int xxx, i;
  unsigned char ccc;
  int err;
  unsigned short *p;
  int fd;
  
  fd = atoi(*(++argv)); 


  /* Alarm for timeout */
  intercept(alarm_hand);

  data=malloc(96);
  /* INIT DATA MODULES AND BUFFERS */

  /* Init Status Data Module */
  if(dm_init(StatModName, sizeof(STATUS), (void **)&StData, &StHeader)) {
    if(Debug) printf("Can't init Status Data Module\n");
    exit(-1);
  }
      
  /* Init Config Data Module */
  if(dm_init(ConfigModName, sizeof(CONFIG), (void **)&CfData, &CfHeader)) {
    if(Debug) printf("Can't init Config Data Module\n");
                exit(-1);
  }
     
  /* Init Event Buffer */
  while(( err = EvbInit(&evbid, EVT_BUFFER_NAME, EVBSIZE)) == E_KWNMOD);
  if( err) {
    if(Debug) printf("Can't init Event Buffer\nError: %d\n", err);
    exit(-1);
  }
  
  /* Init Muon Buffer */
  while(( err = EvbInit(&mubid, MU_BUFFER_NAME, MUBSIZE)) == E_KWNMOD);
  if( err) {
    if(Debug) printf("Can't init Muon Buffer\nError: %d\n", err);
    exit(-1);
  }
 
  /* Init Monitor Buffer */
  while(( err = EvbInit(&MonitId, MONIT_BUFFER_NAME, MONIT_SIZE)) == E_KWNMOD);
  if( err) {
    if(Debug) printf("Can't init Monitor Buffer\nError: %d\n", err);
    exit(-1);
  }
 
  /* REDIRECT STDIN & STDOUT TO THE SOCKET */
  if(_os_close(0) != 0) {
    if(Debug) printf("Can't close stdin\n");
    exit(-1);
  }
  if(_os_dup(fd, &path) != 0) {
                if(Debug) printf("Can't dup socket I\n");
                exit(-1);
        }
  if(_os_close(1) != 0) {
                if(Debug) printf("Can't close stdout\n");
                exit(-1);
        }       
  if(_os_dup(fd, &path) != 0) {
                if(Debug) printf("Can't dup socket II\n");
                exit(-1);
        }
  if(_os_close(fd) != 0) {
                if(Debug) printf("Can't close socket\n");
                exit(-1);
        }       

  /* READ A COMMAND AND SEND THE ANSWER */
  for(;;) {            /*infinite loop*/

    /* Set the timeout alarm */
    if ((err = _os_alarm_set(&alarm_id, 2, TIME(secs))) != SUCCESS) {
      if(Debug) fprintf(stderr,"Can't set alarm. Error: %d\n", err);
      exit(-1);
    }

    /* Get a command */
    if (read_data(0, &command, 1) != 1) {
      /* read_data() should return the number of bytes requested */
      /* If not, there is something wrong. */
      /* Check if we had a timeout */
      if(TimeOut == 1) {
	flag = TOUT;
	number = 0;
	Send_Head(flag, htons(number));
        if(Debug) fprintf(stderr,"Time out\n");
	exit(1);
      } else {
        /* This is really bad */
        if(Debug) fprintf(stderr,"READ COMMAND: Read returned error\n");
        exit(-1);
      }
    }

    /* Send the answer */
    switch(command) {
      
      case CONNECT:    
      /* Send OK:4:01234567 */
        flag = OK;
        number = 4;
        Send_Head(flag, htons(number));
        if(write(1, &test, 4)!=4) exit(-1);
        break;
      
      case EVTBUF:    
      /* Send OK:size:data if found and OK:00 if not */
        err = EvbGetLastTag(evbid, data, &nnnn, TAG_DATA_SVR, 0, 1) ;
        if (err == EVB_ERR_NOT_FOUND) {
          flag = OK;
          number = 0;
          Send_Head(flag, htons(number));
          break;
        } else if ( err != 0 ) {
          exit(-1);
        } else { 
          flag = OK;
          number = nnnn;
          Send_Head(flag, htons(number));
          if(write_data(1, data, number)!=number) exit(-1);
	  break;
        }
      
      case MUONBUF:    
      /* Send OK:size:data if found and OK:00 if not */
        err = EvbGetLastTag(mubid, data, &nnnn, TAG_DATA_SVR, 0, 1) ;
        if (err == EVB_ERR_NOT_FOUND) {
          flag = OK;
          number = 0;
          Send_Head(flag, htons(number));
          break;
        } else if ( err != 0 ) {
          exit(-1);
        } else { 
          flag = OK;
          number = nnnn;
          Send_Head(flag, htons(number));
          if(write_data(1, data, number)!=number) exit(-1);
                                  break;
        }
      
      case MONITBUF:    
      /* Send OK:size:data if found and OK:00 if not */
        err = EvbGetLastTag(MonitId, data, &nnnn, TAG_DATA_SVR, 0, 1) ;
        if (err == EVB_ERR_NOT_FOUND) {
          flag = OK;
          number = 0;
          Send_Head(flag, htons(number));
          break;
        } else if ( err != 0 ) {
          exit(-1);
        } else { 
          flag = OK;
          number = nnnn;
          Send_Head(flag, htons(number));
          if(write_data(1, data, number)!=number) exit(-1);
                                  break;
        }
      
      case DSTATUS:    
      /* Send OK:size:data */
        flag = OK;
        number = sizeof(STATUS);
        Send_Head(flag, htons(number));
        if(write_data(1, (unsigned char *)StData, 
            number)!=number) exit(-1);
        break;
      
      case DCALIB:            
      /* NOT YET IMPLEMENTED Returns OK:00 */
        flag = OK;
        number = 0;
        Send_Head(flag, htons(number));
        break;
      
      case DISCONNECT:  
      /* Send OK:00 and exit */
        flag = OK;
        number = 0;
        Send_Head(flag, htons(number));
        free(data);
	exit(0);
        break;
      
      case NOP:    
      /* Send OK:00 */
        flag = OK;
        number = 0;
        Send_Head(flag, htons(number));
        break;
      
      case LOOKSTAT:    
      /* Send OK:36:(8int) */
        flag = OK;
	number = 36;
	Send_Head(flag, htons(number));
        xxx = htonl(StData->RunStatus);
        if(write_data(1, (unsigned char *)&xxx, 4)!=4) exit(-1);
        xxx = htonl(StData->inter.SlowSec);
        if(write_data(1, (unsigned char *)&xxx, 4)!=4) exit(-1);
        xxx = htonl(StData->inter.T1Fast0 + StData->inter.T1Fast1);
        if(write_data(1, (unsigned char *)&xxx, 4)!=4) exit(-1);
        xxx = htonl(StData->inter.T1Slow);
        if(write_data(1, (unsigned char *)&xxx, 4)!=4) exit(-1);
        xxx = htonl(StData->trg2.T2);
        if(write_data(1, (unsigned char *)&xxx, 4)!=4) exit(-1);
        xxx = htonl(StData->evtsvr.T3Asked);
        if(write_data(1, (unsigned char *)&xxx, 4)!=4) exit(-1);
        xxx = htonl(StData->evtsvr.T3Found);
        if(write_data(1, (unsigned char *)&xxx, 4)!=4) exit(-1);
        xxx = htonl(StData->evtsvr.T3Muon);
        if(write_data(1, (unsigned char *)&xxx, 4)!=4) exit(-1);
        xxx = htonl(StData->evtsvr.T3Lost + StData->evtsvr.T3NotFound);
        if(write_data(1, (unsigned char *)&xxx, 4)!=4) exit(-1);
        break;
      
      case LOOKMON:    
      /* Send OK:96:(3 ints, 42 shorts) */
	err = EvbGetLastTag(MonitId, data, &nnnn, TAG_DATA_SVR, 0, 1) ;
	if (err == EVB_ERR_NOT_FOUND) {
	  flag = OK;
	  number = 0;
	  Send_Head(flag, htons(number));
	  break;
	} else if ( err != SUCCESS ) {
	  exit(-1);
	} else {
	  flag = OK;
	  number = nnnn;
	  Send_Head(flag, htons(number));
	  for(i=0;i<3;i++) {
	      xxx = htonl(*(((int *)data)+i));
	      if(write_data(1, (unsigned char *)&xxx, 4)!=4) exit(-1);
	  }
	  for(i=0;i<42;i++) {
	      xxx = htons(*(((unsigned short *)data)+6+i));
	      if(write_data(1, (unsigned char *)&xxx, 2)!=2) exit(-1);
	  }          
	  break;
	}
      
      case LOOKMONPK:    
      /* PACKED version of LOOKMON */
      /* Send OK:60:(60char) */
        err = EvbGetLastTag(MonitId, data, &nnnn, TAG_DATA_SVR, 0, 1) ;
        if (err == EVB_ERR_NOT_FOUND) {
	  flag = OK;
	  number = 0;
	  Send_Head(flag, htons(number));
	  break;
	} else if ( err != SUCCESS ) {
	    exit(-1);
	  } else {
	      flag = OK;
	      number = 60;
	      Send_Head(flag, htons(number));
              if(write_data(1, data, 8)!=8) exit(-1);
              data = data + 8;
              number = pack((unsigned short *)data, 32, datac);
              if(Debug) fprintf(stderr,"%d\n", number);
	      if(write_data(1, datac, number)!=number) exit(-1);
              data = data + 64;
              if(write_data(1, data, 4)!=4) exit(-1);
              data = data - 72;
              break;
            }
      
      case CHANGEDA:
        /* The client is sending 9 unsigned shorts for D/As */
        if (read_data(0, data, 18) != 18) {
        /* read_data() should return the number of bytes requested */
	  if(Debug) fprintf(stderr,"CHANGEDA: Read returned error\n");
	  exit(-1);
	}
        datash = (unsigned short *)data;
        pconfig= &CfData->MonitParams;
        if(Debug) {
          fprintf(stderr,"New values for DACs:\n");
          for(i=0;i<9;i++) {
            fprintf(stderr,"%d ", datash[i]);
          }
          fprintf(stderr,"\n");
        }
        for( i=0; i<8; i++) pconfig->Dac[i]= ntohs(datash[i]);
        pconfig->OutputRegister= ntohs(datash[8]);
	StData->IsChangeDac = 1;
        break;
      
      default:    
      /* Send NOK:00 */
        flag = NOK;
        number = 0;
        Send_Head(flag, htons(number));
        break;
        
	} /* End of switch */

    /* If I get here, the alarm didn't expire */
    /* So, delete it */
    _os_alarm_delete(alarm_id);

  } /* End od infinite loop */

} /* End of main */



/*****************************************************************************/
void Send_Head(
  unsigned char theflag,    /* the flag */
  unsigned short thenumber  /* the number of bytes */
    )
{
  if(write_data(1, &theflag, 1)!=1) exit(-1);
  if(write_data(1, (unsigned char *)&thenumber, 2)!=2) exit(-1);
}
  
/*****************************************************************************/
int read_data(
    int s,       /* connected socket */
    unsigned char *buf,   /* pointer to the buffer */
    int n          /* number of bytes we want to read */
    )
{ 
  int bcount;         /* counts bytes read */
  int br;             /* bytes read this pass */

  bcount= 0;
  br= 0;
  while (bcount < n) {               /* loop until full buffer */
    if ((br = read(s,buf,n-bcount)) > 0) {
      bcount += br;                  /* increment byte counter */
      buf += br;                     /* move buffer ptr for next read */
    }
    else if (br < 0)                 /* signal an error to the caller */
      return(-1);
    if (TimeOut == 1) return 0;
  }
  return(bcount);
}

/*****************************************************************************/
int write_data(
    int s,          /* connected socket */
    unsigned char *buf,   /* pointer to the buffer */
    int n          /* number of bytes we want to write */
    )
{ 
  int bcount;         /* counts bytes writen */
  int br;             /* bytes writen this pass */

  bcount= 0;
  br= 0;
  while (bcount < n) {               /* loop until full buffer */
    if ((br= write(s,buf,n-bcount)) > 0) {
      bcount += br;                  /* increment byte counter */
      buf += br;                     /* move buffer ptr for next write */
    }
    else if (br < 0)                 /* signal an error to the caller */
      return(-1);
    if (TimeOut == 1) return 0;
  }
  return(bcount);
}


/*****************************************************************************
Pack Function
OVERVIEW:
This function packs 12 bits unsigned shorts, removing the leading 0
nibble. It receives three parameters:

  *datain    a pointer to the shorts to pack
  n    the number of shorts to pack
  *dataout  a pointer where to store the packed bytes

FORMAT:
Two shorts are packed to three chars. The function puts in first place
the less significant byte of the first short, then an OR of the two most
significant bytes (the first one shifted 4 times), and finally puts
the less significant byte of the second short.
For example, if you have two 12 bits shorts 0x0FED and 0x0123, this 
function packs them to three bytes 0xED 0xF1 and 0x23.

  0FED:0123 -> ED:F1:23

RETURN VALUE:
On success, it returns the number of packed bytes. On error, it returns -1.
****************************************************************************/

int pack(unsigned short *datain, int n, unsigned char *dataout) 
{ 
    int i, j, nnn;
    unsigned char *datai;   
    unsigned char tmp;

    datai = (unsigned char *)datain;
    nnn =0;

    i = 1;
    for(j=0;j<2*n;j++) {  
      if(i==1) {tmp = *datai++ << 4; }
      if(i==2) {*dataout++ = *datai++; nnn++;}
      if(i==3) {*dataout++ = tmp | *datai++; nnn++;}
      if(i==4) {*dataout++ = *datai++; nnn++; i=0;}
      i++;
    }
    if(i==3) { *dataout++ = tmp; nnn++; }
    
    return(nnn);

}

/****************************************************************************
Unpack Function
OVERVIEW:
This function unpacks data packed with function pack(), to unsigned shorts.
It receives three parameters:

  *datain    a pointer to the bytes to unpack
  n    the number of bytes to unpack
  *dataout  a pointer where to store the unpacked shorts

FORMAT:
See pack() function.

RETURN VALUE:
On success, it returns the number of unpacked shorts. On error, it returns -1.
****************************************************************************/

int unpack(unsigned char *datain, int n, unsigned short *dataout) 
{
    int i, j, nnn;
    unsigned char *datao;  
    unsigned char tmp, tmp2;

    datao = (unsigned char *)dataout;
    nnn =0;

    i = 1;
    for(j=0;j<n;j++) {
      if(i==1) {tmp = *datain++; }
      if(i==2) {tmp2 = *datain++; *datao++ = tmp2 >> 4; *datao++ = tmp; nnn++; nnn++;}
      if(i==3) {*datao++ = tmp2 & 0x0F; nnn++; *datao++ = *datain++; nnn++; i=0;}
      i++;
    }
    
    return(nnn);
    
}

