/*

	tcpsvr.c
	AV, oct 1999

	This program starts a TCP/IP server listening on port 12345 for incoming
	connections from clients. When it receives one, launches tcpdatasvr.
	Then, it starts waiting for new connectios.

	NOTE:
		To set up a standard BSD socket server, allways:
		((1)) Create a socket
		((2)) Bind an address to that socket
		((3)) Create a listen queue at the socket
		((4)) Accept multiple connections (fork a child for each two)

	This program can be started with the -d option, to enable some debugging
	information to be sent to the standard output.

*/

#include <stdio.h>
#include <process.h>
#include <errno.h>
#include <cglob.h>
#include <INET/socket.h>
#include <INET/in.h>
#include "tcpsvr.h"


/* Func to start a child */
process_id start_child(int);

/* Default is  printfs */
int Debug = 1;

main(int argc, char **argv) {

	int sock, newsock;
	int port = SERVER_PORT;
	struct sockaddr_in sock_name = {AF_INET};
	struct sockaddr_in from;
	int namelength;
      	process_id sonpid ;
	
	/* Handle command line options (only -d: enable printfs) */
	while((argc > 1) && (argv[1][0] == '-')) {
        	switch(argv[1][1]) {
            		case '?':
				printf("TCP server usage: tcpsvr [-d]\n");
                                printf("\t-d  : Enable Debugging\n");
                                exit(0);
                		break;
            		case 'd':
				printf("Debugging enabled\n");
				Debug = 1;
                		break;
            		default:
				Debug = 0;
                		break;
        	}
    		argc--;
   		argv++;
    	}

	/* 
		((1)) 
		GET A SOCKET in sock (gets file descriptor) 
		-------------------------------------------
	*/
	if ((sock = socket (AF_INET, SOCK_STREAM, 0)) == -1) {
		perror ("Socket");
 		exit(0);
	}
	if(Debug) printf("Socket opened OK: %d\n", sock);

	/* 
		((2)) 
		BIND AN ADDRESS TO SOCKET in sock_name, until it works! 
		-------------------------------------------------------
	*/
	if(Debug) printf("Calling BIND\n");
	sock_name.sin_port = htons(port);
	if ( bind( sock, (struct sockaddr *)&sock_name, sizeof (sock_name)) != 0 ) {
		printf( "Bind Error...\n" );
		while (bind(sock, (struct sockaddr *)&sock_name, 
					sizeof (sock_name)) != 0) {
			printf("Bind error again...\n");
			sleep(2);
		}
	}
	if(Debug) printf("Socket binded OK\n");

  	/* 
		((3)) 
		LISTEN TO THE SOCKET for incoming connections 
		---------------------------------------------
	*/
  	if ( listen( sock, 1 ) < 0 ) {
    		perror ("Listen");
    		cleanup(sock);
  	}
    	if(Debug) printf("Listen on port %d OK\n", port);

    	namelength = sizeof(struct sockaddr_in);

  	/* 
		((4)) 
		ACCEPT CONNECTIONS in newsock file descriptor 
		---------------------------------------------
	*/
  	for(;;) {
		if(Debug) printf("Waiting for a new connection on port %d...\n", port);
    		if((newsock = accept(sock, (struct sockaddr *)&from, 
					&namelength)) == -1) {
      			perror("Accept");
      			cleanup(sock);
    		}
    		else {
      			if(Debug) { 
			    printf("\tConnection from host %s, port %d, socket %d\n",
			    inet_ntoa(from.sin_addr), ntohs(from.sin_port), newsock);
      			    printf("\tLaunching \"tcpdatasvr\" for client %s\n", 
			    inet_ntoa(from.sin_addr));
			}
      			sonpid = start_child(newsock) ;
			if(Debug) printf("\tPID for \"tcpdatasvr\": %d\n", sonpid);
    		}
  	}

  	cleanup(sock);	/* I will never get here */
}

/**************************************/
/* This function starts a child       */
/**************************************/
process_id start_child(int fd) {
	
	process_id pid;
	u_int16 type_lang;
	char *argblk[6];
	
	type_lang = mktypelang(MT_PROGRAM, ML_OBJECT);
	
	argblk[0] = "tcpdatasvr";
        sprintf(argblk[1], "%i", fd);
	argblk[2] = (char *) 0;
	
	if((errno = _os_exec( _os_fork, 0, 5, argblk[0], argblk, _environ, 
			0, &pid, type_lang, 1)) != 0)
		exit(_errmsg(errno, "Problem creating child %s\n", argblk[0]));
	
	wait(0);
	close(fd);
	return pid;

}


/********************************/
/* Exit gracefully              */
/********************************/
cleanup(int sock) {

	if(shutdown(sock, 2) == -1) perror("shutdown");
	exit(0);
}

