/* 
   trigger2 : Local Station Software - Pierre Auger Project
   ----------
              JM Brunet, L Gugliemi, C Le Gall, G Tristram
	               College de France - Paris

   Created    01/06/97 - JMB
   Modified : 13/10/97 - Send all T2 contained in one second
              16/11/98 - Version 2
	      02/03/98 - Version 2.2:Data sent to Central Station for each
	                             Trigger 2 (20 bits) : 
				     -  4 bits : energy evaluation
				     - 20 bits : micro second of the 1 part.

----------------------------------------------------------------------------*/
#include <time.h>
#include <errno.h>

#define MAIN
#define _CONFIG_MAIN_
#define MYNAME  TRIGGER2

#include "ready_acq.h"
#include "msgdefs.h"
#include "dmlib.h"
#include "asiclib.h"

#include "augererr.h"

#include "central.h"
#define _STATUS_MAIN_
#include "status.h"
#include "evbdefs.h"
#include "run_config.h"

int VerticalEquivalentMuon= 100;   /* WARNING : this value must be taken
				      from CalMonit data module */

int TotalSig=0, Debug= 1;
signal_code Signal = 0 ;
char StrLog[256];

STATUS *st;
TRIG1  *t1;
CONFIG *cf;

PROCESS *Pproc;

extern void *_glob_data;

void  init( void);
void  DoWork( void);
int   trigger2( FAST_EVENT *);
int   T2Algo( FAST_EVENT * , u_int32 *);
void  byebye( error_code);
int   ShowFastBuffer( FAST_EVENT *pbuf);

void
sig_hand( signal_code s )
{
  Signal= s;
  if( s == Pproc->notify->param1) TotalSig++;
  if( s != SIG_PAUSE && s != SIG_STOP) byebye( 30000);
  _os_rte();
}





main( int argc, char *argv)
{
  
  init();
  DoWork();
  byebye( 4000);
}

void DoWork( void)
{
  int32 value, min_val=1, max_val=0x1000;
  signal_code signal;
  error_code err;
  FAST_EVENT *pfast, *pevt;
  int n, size;
  unsigned char tag[]= {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};

  while( 1) {             /* Infinite loop : Wait for an event from ReadOut */
    err=_os_ev_wait( st->event.IdT1Trigger2, &value, &signal, 
		     min_val, max_val);
    if( value == 1 && st->RunStatus == REBOOT) break;
    if( Signal == SIG_STOP) {            /* Stop Run */
      if( (err= EvbReset(evbid))) {       /* Reset Event Buffer */
	sprintf( StrLog, "tigger2 : error %d calling EvbReset()\n", err);
	PrtLog( StrLog, 1);
      }
      Signal= 0; continue;
    }
    if( Signal == SIG_PAUSE) {           /* Pause Run */
      trigger2( NULL);                   /* Send remaining T2s to CS */
      Signal= 0; continue;
    }
    if( err) break;
                                         /* Ask for the oldest Fast Buffer */
    if(( err= FB_use( &pfast))) {
      sprintf( StrLog, "trigger2: Error %d calling FB_use()\n", err);
      PrtLog( StrLog, 1);
      break;
    }
    if( pfast->T2T1_type & 1) st->trg2.T1Fast0++;
    else st->trg2.T1Fast1++;
                                         /* Event Buffer Allocation */
    while( err = EvbAlloc( evbid, (void **)&pevt, sizeof(FAST_EVENT))) {
      if( err != EVB_ERR_FULL) {
	sprintf( StrLog, "trigger2: error $%x in EvbAlloc\n",err);
	PrtLog( StrLog, 1);
	break;
      }
      if( err = EvbFree( evbid, NB_EVB_RELEASE)) {
	sprintf( StrLog, "trigger2: error $%x in EvbFree\n",err);
	PrtLog( StrLog, 1);
	break;
      }
      st->trg2.NEvtBusy -= NB_EVB_RELEASE;
      if( st->trg2.NEvtBusy <= 0) {
	PrtLog( "trigger2 : ????????????????????\n", 1);
	goto wrong;
      }
      st->buffer.EvbUsed -= NB_EVB_RELEASE;
    }
    *pevt=*pfast;                    /* Copy RawEvtBuff into EvtBuffer */
    if(( err= FB_free())) {          /* Release Fast buffer */
      sprintf( StrLog, "trigger2: Error %d calling FB_free()\n", err);
      PrtLog( StrLog, 1);
      break;
    }
    trigger2( pevt);                 /* Is it a T2 event ? */
                                     /* Compute the size of the event */
    size= sizeof( TimeStamp) + 
      sizeof( TimeStamp)     + 
      sizeof( unsigned int)  +
      sizeof( unsigned short)+
      sizeof( unsigned short)+
      pevt->nsamples*sizeof(FAST_SAMPLE);
                                     /* Mark Event buffer as ready */
                                     /* and shrink it to size */
    if( err= EvbReady( evbid, (void *)pevt, size, tag)) {
      sprintf( StrLog, "trigger2: EvbReady error %d\n", err);
      PrtLog( StrLog, 1);
      break;
    }
                                     /* Event buffer occupancy */
    st->buffer.FastUsed--;
    st->buffer.EvbUsed++;
    st->trg2.NEvtBusy++;
    if( st->trg2.T1Fast0+st->trg2.T1Fast1 > 1000) {
      n= st->trg2.NEvtBusy;
      if( n > st->buffer.EvbMax) st->buffer.EvbMax= n;
      if( n < st->buffer.EvbMin) st->buffer.EvbMin= n;
    } 
    if( st->buffer.EvbMin > st->buffer.EvbUsed) 
      st->buffer.EvbMin= st->buffer.EvbUsed;
    if( st->buffer.EvbMax < st->buffer.EvbUsed) 
      st->buffer.EvbMax= st->buffer.EvbUsed;
  }
 wrong:
  sprintf( StrLog, "trigger2: EndLoop error= %d\n", err);
  PrtLog( StrLog, 1);
}


void init( void)
{
  error_code err;
  u_int32 perm= 0x777;
  signal_code sig;

                                        /* Link to Data Modules        */
  if( dm_init( StatModName, sizeof( STATUS), (void **)&StData, &StHeader))
    exit( _errmsg( errno, "trigger2 : Can't create STATUS data module!\n"));
  if( dm_init( ConfigModName, sizeof( CONFIG), (void **)&CfData, &CfHeader))
    exit( _errmsg( errno, "trigger2 : Can't create CONFIG data module!\n"));
  if( dm_init( T1ModName, sizeof( TRIG1), (void **)&T1Data, &T1Header))
    exit( _errmsg( errno, "trigger2 : Can't create TRIGGER1 data module!\n"));
  st= StData;
  t1= T1Data;
  cf= CfData;
                                        /* Process pointer (msgdefs.h) */
  if(( Pproc= myprocess( MYNAME)) == NULL) {
    PrtLog( "trigger2 : error calling myprocess\n", 1);
    exit( 0);
  }
                                        /* Link to T1Tg2 Event */
  if(( err=  _os_ev_link( st->event.EvNameT1Tg2, &st->event.IdT1Trigger2))
     != SUCCESS)
    exit(_errmsg(errno, "trigger2: Can't link to the Fast event!\n"));
                                        /* Init Fast Buffers */
  if(( err= FB_init( MAX_FAST_BUFFER))) {
    sprintf( StrLog, "trigger2: Can't init. fast Buffers: err= %d\n", err);
    PrtLog( StrLog, 1);
    exit( 0);
  }
                                        /* Event Buffer initialisation */
  while(( err= EvbInit(&evbid, EVT_BUFFER_NAME, EVBSIZE)) ==  E_KWNMOD);
  if( err) {
    sprintf( StrLog, "trigger2: Error $%x EvbInit\n", err);
    PrtLog( StrLog, 1);
    byebye( err);
  } 
  if(( err= ReadyInit( &ReadyEvId, ReadyEvName))) {
    sprintf( StrLog, "trigger2: Error $%x call ReadyInit\n", err);
    PrtLog( StrLog, 1);
    byebye( err);
  }
  if(( err= ReadyToGo( ReadyEvId, READY_TRIGGER2))) {
    sprintf( StrLog, "trigger2: Call ReadyToGo() err=$%x\n", err);
    PrtLog( StrLog, 1);
    byebye( err);
  } 
  printf( "trigger2 : waiting for msgsvrout\n"); fflush( stdout);
                                        /* Wait for msgsvrout (MailBox) */
  if(( err= ReadyAll( ReadyEvId, &sig, READY_MSGSVROUT)) 
     != E_SUCCESS) {
    sprintf( StrLog, "trigger2: Unexpect sig %d in ReadyAll() : err= %d\n", 
	     sig, err);
    PrtLog( StrLog, 1);
    byebye( err);
  }
  printf( "trigger2 : Open Mailbox in send mode\n"); fflush( stdout);
                                        /* Open Mailbox in send mode */
  if((err = PostLink( &Pproc->mbx_out, Pproc->pnameout ))) {
    sprintf( StrLog, "trigger2: Can't Open Mailbox '%s' for Send: %d\n",
	       Pproc->pnameout , err ) ;
    PrtLog( StrLog, 1);
    byebye( err);
  }
                                        /* Intercept Signals and mask them */
  _os_intercept( sig_hand, _glob_data );
  err= ReadyFinish( ReadyEvId, ReadyEvName);     
  printf( "trigger2 : init ended...\n"); fflush( stdout);
}



void byebye( error_code err)
{
  dm_finish( &StHeader );               /* Unlink Status data module */
  FB_finish();                          /* Unlink Fast buffer */
  EvbFinish(&evbid);                    /* Unlink Event Buffer */
  PostClose( &Pproc->mbx_out);          /* Close Mailbox */
                                        /* Unlink + delete OS9-event */
  while( _os_ev_unlink( st->event.IdT1Trigger2));
  _os_ev_delete( st->event.EvNameT1Tg2);
  if( Signal) {
    sprintf( StrLog, "trigger2 : ended by signal %d\n", Signal);
    PrtLog( StrLog, 1);
    exit(_errmsg(errno, "trigger2 : ended...\n"));
  }
}


int trigger2( FAST_EVENT *pevt)
{
  int j,s=0;
  static MESSAGE *pmes;
  static int size_t2_yes= 0, oldsec= 0;
  error_code err;
  u_int32 Enano, t;
  signal_code dummy;
  static unsigned char *mesbuf;
  static int IsAlloc= 0;
  
  if( pevt == NULL) {        /* Pause run requested */
    if( IsAlloc) {
      pmes->mheader.length= size_t2_yes;
      PostSendShrink(  Pproc->mbx_out, pmes, size_t2_yes);
    }
    size_t2_yes= 0;
    IsAlloc= 0; return 0;
  }
                                      /* Compute E, nano & T2 algorithm */
  if( !T2Algo( pevt, &Enano)) return 0;
  if( !IsAlloc) {            /* Allocate a new buffer in the mailbox */
    while(( err= PostAlloc( Pproc->mbx_out, (void **)&pmes, 
			    Pproc->Bsize_out)) != E_SUCCESS) {
      if( err == POST_ERR_FULL) {
	t= 1; _os_sleep( &t, &dummy); /* Mailbox full, try again */
      }
    }
    IsAlloc= 1;
    oldsec= pevt->datep.second;
    mesbuf= pmes->buf;
    pmes->mheader.type= M_T2_YES;
    pmes->mheader.version =0;
    *(int *) mesbuf= oldsec;
    mesbuf += sizeof( oldsec);
                                      /* header + second         */
    size_t2_yes= sizeof( MESSAGE_HEADER)+ sizeof( oldsec);
  }
  pevt->T2T1_type |= 0x2000;
  st->trg2.T2++;
	                              /* If new second, send all previous T2 
			                 and allocate a new mailbox buffer */
  if( pevt->datep.second != oldsec) {
    st->trg2.second= pevt->datep.second;
    if( size_t2_yes != 0) {
      pmes->mheader.length= size_t2_yes;
      if((err=PostSendShrink(  Pproc->mbx_out, pmes, size_t2_yes))
	 != E_SUCCESS) {
	sprintf( StrLog, "t2yes: PostSendShrink error= %d\n", err);
	PrtLog( StrLog, 1);
      }
      while(( err= PostAlloc( Pproc->mbx_out, (void **)&pmes, 
			      Pproc->Bsize_out)) != E_SUCCESS) {
	if( err == POST_ERR_FULL) {
	  t= 1; _os_sleep( &t, &dummy); /* Mailbox full, try again */
	}
      }
    }
    IsAlloc= 1;                       /* New buffer : store second */
    oldsec= pevt->datep.second;
    mesbuf= pmes->buf;
    pmes->mheader.type= M_T2_YES;
    *(int *) mesbuf= oldsec;
    mesbuf += sizeof( oldsec);
                                      /* header + second         */
    size_t2_yes= sizeof( MESSAGE_HEADER)+ sizeof( oldsec);
  }
  *mesbuf++= (Enano >> 16) & 0xFF;   /* Store Eval & Micro : 24 bits */
  *mesbuf++= (Enano >>  8) & 0xFF;
  *mesbuf++=  Enano        & 0xFF;
  size_t2_yes += 3;
  return 1;
}

int T2Algo( FAST_EVENT *pfast, u_int32 *Enano)
{
 /* search for :
      - First particule micro second
      - Energy evaluation : number of Vertical Equivalent Muons
    trigger2 algorithm (return value)
 */  
  FAST_SAMPLE *pdata= pfast->data;
  unsigned short nwords, n;
  unsigned int TimeTagTick;
  unsigned int TickRange, PmtSum, thresh, nano, eval;
  u_int32 micro=0;
  int tick, offset= 0;
  char IsFast0, IsFast1;
  static int prev_sec=0;
  static unsigned char *pbuf= BufCom;
  unsigned int Energy;
  FAST_SAMPLE *pd;
   
  pfast->datep.second = pfast->date0.second; 
  pfast->datep.nano   = pfast->date0.nano; 
  nwords= pfast->nsamples;
  TimeTagTick= (pfast->time_tag & 0xFFFF);
  IsFast0= (pfast->T2T1_type & 0x4);
  IsFast1= (pfast->T2T1_type & 0x8);
  if( IsFast0) {                      /* Total tick range */
    TickRange= cf->FeParams.fast0.Delay+ cf->FeParams.fast0.Width;
    thresh= cf->FeParams.fast0.Thresh;
  } else {
    TickRange= cf->FeParams.fast1.Delay+ cf->FeParams.fast1.Width;
    thresh= cf->FeParams.fast1.Thresh;
  }
                                      /* First tick in window */
  tick= TimeTagTick- TickRange;
  if( tick < 0) { 
    offset= -tick; tick= 0;
  }
                                      /* First shower particle TimeStamp */
  for( n=0; n<nwords; n++, pdata++) {
  if( ((pdata->time_tag+ offset) & 0xffff) < tick) continue;
    PmtSum= pdata->pmt1+ pdata->pmt2+ pdata->pmt3;
    if( PmtSum < thresh) continue;
    tick= TimeTagTick- pdata->time_tag;
    if( tick < 0) tick += 0x10000;
                                      /* New TimeStamp */
    nano= (tick << 1) + (tick >>1);   /* nano= tick*2.5 */
    if( pfast->datep.nano < nano) {
      pfast->datep.second -=1;
      pfast->datep.nano +=  100000000;
    }
    pfast->datep.nano -= nano;
    micro= pfast->datep.nano/100;
    break;
  }
  pfast->datep.nano= micro;
  pd= pdata;                          /* Energie evaluation */
  nwords -= n;
  Energy= 0;
  for( n=0; n<nwords; n++, pd++) {
    if( pd->lowd8)  Energy += (pd->lowd8 << 6);
    else if( pd->low)  Energy += (pd->low << 3);
    else Energy += pd->sumd8;
  }
  eval= 0;
  if( Energy) eval= Energy/VerticalEquivalentMuon;
  if( eval > cf->AcqParams.VEMmin) eval -= cf->AcqParams.VEMmin;
  if( eval > 15) eval=15;
  *Enano= (eval << 20) | (micro & 0xFFFFF);
                                      /* Fake Trigger2 Algorithm */
  if( !t1->ratet2) return 0;
  switch( cf->AcqParams.Trigger2Algorithm) {
  case 0:                             /* Random */
    if( !prob( t1->rate/t1->ratet2)) return 0;
    else return 1;
  default:                            /* ?????? */
    return 0;
  }
}




int ShowFastBuffer( FAST_EVENT *pbuf)
{
  int n;

  printf( "           *****    Fast   Buffer     *****\n");
  printf( "           ********************************\n\n");
  printf( "TimeStamp : second= %d ($%X) - nano= $%x\n", 
	  pbuf->date0.second, pbuf->datep.second, pbuf->datep.nano);
  printf( "TimeStamp : second= %d ($%X) - nano= $%x\n", 
	  pbuf->date0.second, pbuf->date0.second, pbuf->date0.nano);
  printf( "TimeTag   : %d\n", pbuf->time_tag);
  printf( "Trigger Status : $%X\n", pbuf->T2T1_type);
  printf( "Sample number  : %d\n", pbuf->nsamples);
  for( n=0; n<5; n++) {
    printf( "%6d - ", n);
    printf( "%02X%02X%02X%02X ", 
	    pbuf->data[n].sumd8, pbuf->data[n].pmt3,
	    pbuf->data[n].pmt2, pbuf->data[n].pmt1);
    printf( "%04X%02X%02X\n", 
	    pbuf->data[n].time_tag,
	    pbuf->data[n].lowd8, pbuf->data[n].low);
  }
}

