/******************************************************************

  Le buffer des evenements est un DataModule.
  On a la-dedans
    La liste des zones libres.
    La liste ( plus ancien d'abord ) des evenements.
  L'exclusion mutuelle ( reservation et cancel de messages ) se fait via un
  semaphore.

  Les fonctions:
   EVBCreate( char *name, int size, EVBId *id )
      Create the event buffer
   EVBLink( char *name, EVBId *id ) ;
      Link to an existing event buffer
   EVBAlloc( EVBId id, void **buff, int size )
      Request a buffer
   EVBReady( EVBId id, void *buff, int new_size )
      Add the buffer to the list of used buffers and shrink to new size.
   EVBDelete( EVBId id, int nb )
      Delete the nb oldest used buffers
   EVBGetFirstMarked( EVBID id, void **buff, int index, unsigned char mark )
      Get the first buffer with marker[index] == mark
   EVBMarkBuffer( EVBID id, void *buff, int index, unsigned char mark )
      Set the buffer mark marker[index] = mark
   EVBFind( EVBId id, int seconds, int nanos, int delta, void **buff )
      Search a buffer whose time is in the window.

  *************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <semaphore.h>
#include <signal.h>
#include <events.h>
#include <process.h>
#include <errno.h>

#if EVB_VERSION<5
typedef enum {
  TOO_YOUNG,
  IN_TIME,
  TOO_OLD
} IN_DATE ;
#endif

#if defined(OLD_VERSION)
#include "auger_err.h"
#else
#include "augererr.h"
#include "augerstrerr.h"
#endif
#include "evb_defs.h"
#include "evblib.h"

#define EVB_GOOD_BUF 0xDADADEDE

static char Version[64] ;

static error_code
evb_ok( evbuffer *evb )
{
  if ( !evb || evb->synch != EVB_SYNCH )
    return EVB_ERR_UNKNOWN ;
  return SUCCESS ;
}

static error_code
evb_mine( evbuffer *evb )
{
  if ( evb->owner != my_pid() ) return EVB_ERR_NOTYOURS ;
  return SUCCESS ;
}

static error_code
evb_create( char *mname, int size, void **head, mh_com **addr )
{
  unsigned short attrev = 0xC000, /* Sharable, Sticky Bit */
    typlang = 0x400; /* Type "data module" */
  unsigned long perm = 0x333; /* RW pour tout le monde */
  error_code ok ;

  ok = _os_datmod(
		  mname,
		  size,
		  &attrev,
		  &typlang,
		  perm,
		  head,
		  addr) ;
    /* the data module has just been created */
    return ok ;
}

static error_code
evb_link(  char *mname, void **head, mh_com **addr )
{
  unsigned short attrev = 0xC000, /* Sharable, Sticky Bit */
    typlang = 0x400; /* Type "data module" */
  unsigned long perm = 0x333; /* RW pour tout le monde */
  error_code ok ;

  ok = _os_link(&mname,
		addr,
		head,
		&typlang,
		&attrev ) ;
  /* the data-module already exists */
  return ok ;
}

static error_code
evb_finish( EvbId *evbid, mh_com *addr )
{
  if ( *evbid ) _os_unlink ( addr ) ;
  *evbid = 0 ;
  return 0 ;
}

static error_code
evb_lock( semaphore *sem )
{
  return _os_sema_p( sem ) ;
}

static error_code
evb_unlock( semaphore *sem )
{
  return _os_sema_v( sem ) ;
}

/***************************************************
  Gestion des listes chainees

  evb_unlink
  evb_tail
  evb_head
  evb_before
  **************************************************/

static void
evb_tail( evb_buf_header *new, evb_buf_header **first, evb_buf_header **last )
{
  evb_buf_header *old ;
  if ( !(*first) ) {
    *first = *last = new ;
    new->next = new->prev = NULL ;
  }
  else {
    old = *last ;
    new->next = old->next ;
    new->prev = old ;
    old->next = new ;
    *last = new ;
  }
}

static void 
evb_head( evb_buf_header *new,
	   evb_buf_header **first,
	   evb_buf_header **last )
{
  evb_buf_header *old ;
  if ( !(*first ) ) {	/* List is empty */
    new->next = new->prev = NULL ;
    *first = *last = new ;
  }
  else {
    old = *first ;
    new->next = old ;
    new->prev = NULL ;
    old->prev = new ;
    *first = new ;
  }
}

static void
evb_before( evb_buf_header *new,
	    evb_buf_header *old,
	    evb_buf_header **first,
	    evb_buf_header **last )
{
  new->prev = old->prev ;
  old->prev = new ;
  if ( new->prev ) new->prev->next = new ;
  else *first = new ;
  new->next = old ;
}

static void
evb_unlink( evb_buf_header *old,
	     evb_buf_header **first,
	     evb_buf_header **last )
{
  if ( old->next ) old->next->prev = old->prev ;
  else *last = old->prev ;
  if ( old->prev ) old->prev->next = old->next ;
  else *first = old->next ;
}

/*****************************************************
  Gestion de la memoire
  ***************************************************/

static evb_buf_header *
mem_extract( int size, evbuffer *evb )
{
  /* reherche parmi les zones libres une zone de taille suffisante
     et mettre a jour la liste.
     Il faudra y ajouter la glue pour avoir un multiple de 4 bytes */
  evb_buf_header *pbuf = evb->free_head, *qbuf, *nbuf ;
  int bsize, msize ;

  bsize = size + sizeof( evb_buf_header ) ;
  /*  if ( (bsize & 0x3 ) != 0 ) bsize = ( bsize & 0xFFFFFFFC ) + 4 ; */
  msize = bsize + sizeof( evb_buf_header ) + MINIMUM_BUF_SIZE ;
  for( ; pbuf ; pbuf = pbuf->next ) {
    if ( pbuf->bsize >= bsize ) {
      /* trouve un bloc assez grand, y a pluka
	 si taille du bloc trouve - taille du nouveau bloc < minimum alors
	     prendre tout le bloc.
	 sinon creer un bloc de la taille adequate.
	   On prend le nouveau bloc dans la memoire basse de l'ancien */
      if ( pbuf->bsize >= msize ) {
	qbuf = pbuf ;
	pbuf = (evb_buf_header *)((int)qbuf + bsize ) ;
	pbuf->bsize = qbuf->bsize - bsize ;
	/* Insert pbuf */
	evb_before( pbuf, qbuf, &evb->free_head, &evb->free_tail ) ;
	evb_unlink( qbuf, &evb->free_head, &evb->free_tail ) ;
	nbuf = qbuf ;
	nbuf->bsize = bsize ;
      }
      else {
	nbuf = pbuf ;
	evb_unlink( pbuf, &evb->free_head, &evb->free_tail ) ;
      }
      nbuf->buf_ok = EVB_GOOD_BUF ;
      nbuf->size = size ;
      return nbuf ;
    }
  }
  return NULL ;
}

void static
garbage_collect( evbuffer *evb, evb_buf_header *begin, evb_buf_header *end )
{
  evb_buf_header *p ;
  int bsize, success = 0 ;

  /* concatener les blocks libres consecutifs */
  for( p = begin ; p != end ; )
    if ( (unsigned int)p->next == ((unsigned int)p + p->bsize ) ) {
      success = 1;
      bsize = p->next->bsize ;
      if ( p->next == evb->free_tail ) {
	evb->free_tail = p ;
	p->next = NULL ;
	p->bsize += bsize ;
	break ;
      }
      else evb_unlink( p->next, &evb->free_head, &evb->free_tail ) ;
      p->bsize += bsize ;
    }
    else p = p->next ;
  evb->gattempt++ ;
  evb->gsuccess += success ;
}

static void
mem_insert( evb_buf_header *pbuf, evbuffer *evb )
{
  evb_buf_header *p ;

  /* Il faudrait inserer le block dans l'ordre des adresses croissantes
     ce qui permettrait de faciliter la garbage collection */
  pbuf->size = 0 ;
  pbuf->buf_ok = 0 ;
  /* rechercher l'endroit ou inserer ce bloc */
  for( p = evb->free_head ; p != evb->free_tail ; p = p->next )
    if ( p > pbuf ) {
      /* C'est ici qu'il faut l'inserer */
      evb_before( pbuf, p, &evb->free_head, &evb->free_tail ) ;
      return ;
    }
  if ( evb->free_tail < pbuf ) 
    evb_tail( pbuf, &evb->free_head, &evb->free_tail ) ;
  else
    evb_before( pbuf, evb->free_tail, &evb->free_head, &evb->free_tail ) ;
}

/**********************************************************
  Les vraies fonctions
  *********************************************************/

int
EvbVersionId()
{
  return EVB_VERSION ;
}

char *
EvbVersion()
{
  sprintf( Version, "Event Buffer Library V%d ( %s )", 
	   EVB_VERSION, __DATE__ ) ;
  return Version ;
}

char *
EvbError( error_code ok )
{

#if defined(OLD_VERSION)
  if ( ok < EVB_ERR_NO_MEMORY || ok >= EVB_UNKNOWN_ERROR )
    ok = EVB_UNKNOWN_ERROR ;
  return Evb_error_string[ok - EVB_ERR_NO_MEMORY] ;
#else
  return err_comment( (AUGER_ERRORS)ok ) ;
#endif

return NULL;
}

void
EvbDump( EvbId evbid, FILE *f, int part )
{
  evbuffer *pevb = (evbuffer *)evbid ;
  evb_buf_header *pf ;
  error_code ok ;
  int i ;

  if ( ok = evb_ok( pevb ) ) {
    fprintf( f, "%s\n", EvbError( EVB_ERR_UNKNOWN ) ) ;
    return ;
  }
  fprintf( f, ">>>>>>>> Dump du header de la Mailbox\n" ) ;
  fprintf( f, "Adresse de la mailbox : %x\n", pevb ) ;
  fprintf( f, " Vers  = %d\n", pevb->version ) ;
  fprintf( f, " Synch = %x\n", pevb->synch ) ;
  fprintf( f, " Size  = %d\n", pevb->size ) ;
  fprintf( f, " Nbbuf= %d\n", pevb->nbbuf ) ;
  fprintf( f, " Owner = %d\n", pevb->owner ) ;
  fprintf( f, " Garbage collections %d ( %d successful )\n",
	   pevb->gattempt, pevb->gsuccess ) ;
  fprintf( f, " Free Blocks : Head = %x, Tail = %x\n",
	   pevb->free_head, pevb->free_tail ) ;
  if ( part ) for( i=0, pf = pevb->free_head ; pf ; pf = pf->next, i++ )
    fprintf( f, "  Size = %d, addr = %x to %x\n", 
	     pf->bsize, pf, (unsigned int)pf + pf->bsize ) ;
  fprintf( f, " Buf Blocks  : Head = %x, Tail = %x\n",
	   pevb->buf_head, pevb->buf_tail ) ;
  if ( part ) for( i=0, pf = pevb->buf_head ; pf ; pf = pf->next, i++ )
    fprintf( f, "  Size = %d, addr = %x\n", pf->size, pf ) ;

}

static error_code
EvbCreate( EvbId *evbid, char *name, int size )
{
  /* **evb : adress de l'identificateur de la boite evbale.
     *name : pointeur sur le nom de la boite evbale.
     size  : taille de la boite evbale.
     *mode : pointeur sur la strctuure decrivant le mode de notification.

     Code de retour :
       EVB_ERR_ALREADY_EXIST	: la boite existe deja.
       EVB_ERR_NO_MEMORY	: pas de place en memoire pour creer
                                  le data module

				  */
  evbuffer *pevb ;
  mh_com *addr ;
  error_code ok ;

  if ( ok = evb_create( name, size+sizeof( evbuffer ), (void **)&pevb, &addr ) )
    return ok ;

  /* Evbuffer created, now initialize it */
  pevb->version = EVB_VERSION ;
  pevb->owner = my_pid() ;
  pevb->nbbuf = 0 ;
  pevb->size = size ;
  pevb->addr = addr ;
  /* Initialise the semaphore */
  if( ok = _os_sema_init( &pevb->exclusion ) ) {
    evb_finish( evbid, pevb->addr ) ;
    return ok ;
  }
  /* Initialize pointers */
  pevb->memory_arena = ( void *)(pevb+1) ;
  pevb->buf_head = pevb->buf_tail = NULL ;
  pevb->free_head = (evb_buf_header *)(pevb + 1) ;
  pevb->free_tail = pevb->free_head ;
  pevb->free_head->prev = pevb->free_head->next = NULL ;
  pevb->free_head->bsize = (int)pevb->size ;
  pevb->synch = EVB_SYNCH ;
  *evbid = (EvbId)pevb ;
  return SUCCESS ;
}

error_code
EvbFinish( EvbId *evbid )
{
  evbuffer *pevb = (evbuffer *)evbid ;
  error_code ok ;

  if ( ok = evb_ok( pevb ) ) return ok ;
  if ( ok = evb_lock( &pevb->exclusion ) ) return ok ;
  evb_finish( evbid, pevb->addr ) ;
  *evbid = 0 ;
  evb_unlock( &pevb->exclusion ) ;
  return SUCCESS ;
}

error_code
EvbCleanUp( EvbId evbid )
{
  evbuffer *evb = (evbuffer *)evbid ;
  error_code ok ;

  if ( ok = evb_ok( evb ) ) return ok ;
  if ( ok = evb_lock( &evb->exclusion ) ) return ok ;
  garbage_collect( evb, evb->free_head, evb->free_tail ) ;
  return ok = evb_unlock( &evb->exclusion ) ;
}

static error_code
EvbLink( EvbId *evbid, char *name )
{
  /* **evb : adresse de l'identificateur de la boite evbale rendu par la
             fonction.
     *name : pointeur sur le nom.

     Code de retour:
       EVB_ERR_UNKNOWN	: boite evbale inexistante.

       */
  evbuffer *pevb ;
  mh_com *addr ;

  /* if ( CANT link to data module ) then return error
     else if ( NOT an active evbuffer ) then
       unlink data module
       return error
     else return OK
     endif
  */
  if ( evb_link( name, (void **)&pevb, &addr ) )
    return EVB_ERR_UNKNOWN ;
  else {
    if ( pevb->synch != EVB_SYNCH ) {
      evb_finish( evbid, addr ) ;
      return EVB_ERR_UNKNOWN ;
    }
  }
  *evbid = (EvbId)pevb ;
  return SUCCESS ;
}

error_code
EvbInit( EvbId *evbid, char *name, int size )
{
  error_code ok ;

  if ( (ok = EvbLink( evbid, name )) == SUCCESS ) return ok ;
  else return EvbCreate( evbid, name, size ) ;
}

error_code
EvbAlloc( EvbId evbid, void **data, int size )
{
  /*
     *evb ; identificateur du buffer arena.
     **data : adresse du buffer donne par la fonction.
     size   ; taille du buffer demande.

     Code de retour:
       EVB_ERR_UNKNOWN	: le buffer arena n'existe pas ( ou plus !)
       EVB_ERR_FULL	: plus de place.

  */
  error_code ok ;
  evb_buf_header *pdat ;
  evbuffer *evb = (evbuffer *)evbid ;

  if ( ok = evb_ok( evb ) ) return ok ;
  if ( ok = evb_lock( &evb->exclusion ) ) return ok ;;
  if ( ok = evb_ok( evb ) ) {
    evb_unlock( &evb->exclusion ) ;
    return ok ;
  }
  /* Chercher parmi les blocs libres s'il y a de la place */
  pdat = mem_extract( size, evb ) ;
  if ( !pdat ) {
    garbage_collect( evb, evb->free_head, evb->free_tail ) ;
    if ( !(pdat = mem_extract( size, evb )) ) {
      evb_unlock( &evb->exclusion ) ;
      return EVB_ERR_FULL ;
    }
  }
  *data = (void *)((char *)pdat + sizeof( evb_buf_header )) ;
  evb_unlock( &evb->exclusion ) ;
  return SUCCESS ;
}

#if EVB_VERSION<=4

error_code
EvbReady( EvbId evbid, void *pdata, int size )
{
  /* Shrink et rend le buffer accessible */
  evbuffer *evb = (evbuffer *)evbid ;
  evb_buf_header *pbuf, *nbuf ;
  error_code ok ;
  int nsize ;

  if ( ok = evb_ok( evb ) ) return ok ;
  if ( ok = evb_lock( &evb->exclusion ) ) return ok ;

  pbuf = (evb_buf_header *)((char *)pdata - sizeof(evb_buf_header) ) ;
  if ( pbuf->buf_ok != EVB_GOOD_BUF ) {
    evb_unlock( &evb->exclusion ) ;
    return EVB_ERR_BAD_BUF_POINTER ;
  }

  if ( pbuf->bsize > (size + sizeof( evb_buf_header ) + MINIMUM_BUF_SIZE) ) {
    /* On peut shrinker le buffer */
    nbuf = (evb_buf_header *)((char *)pbuf + size + sizeof( evb_buf_header) ) ;
    nsize = pbuf->bsize ;
    pbuf->bsize = size + sizeof( evb_buf_header ) ;
    pbuf->size = size ;
    nbuf->bsize = nsize - ( pbuf->bsize ) ;
    mem_insert( nbuf, evb ) ;
    garbage_collect( evb, nbuf, evb->free_tail ) ;
  }
  /* mettre le buffer en queue de la liste des messages */
  evb_tail( pbuf, &evb->buf_head, &evb->buf_tail ) ;
  evb->nbbuf++ ;
  if ( ok = evb_unlock( &evb->exclusion ) ) return ok ;
  return SUCCESS ;
}
#endif

error_code
EvbFree( EvbId evbid, int nb )
{
  /* Cancels the nb oldest buffers */
  evbuffer *evb = (evbuffer *)evbid ;
  evb_buf_header *pbuf ;
  error_code ok ;
  int i = 0 ;

  if ( (ok = evb_ok( evb )) || (ok = evb_mine( evb )) ) return ok ;
  if ( ok = evb_lock( &evb->exclusion ) ) return ok ;
  if ( nb == 0 ) nb = evb->nbbuf ;
  else if ( evb->nbbuf < nb ) nb = evb->nbbuf ;

  for( ; i < nb ; i++ )
    if ( pbuf = evb->buf_head ) {
      evb_unlink( pbuf, &evb->buf_head, &evb->buf_tail ) ;
      mem_insert( pbuf, evb ) ;
      evb->nbbuf-- ;
    }
  garbage_collect( evb, evb->free_head, evb->free_tail ) ;
  return evb_unlock( &evb->exclusion ) ;
}

error_code
EvbReset( EvbId evbid )
{
  return EvbFree( evbid, 0 ) ;
}

error_code
EvbCancel( EvbId evbid, void *pdata )
{
  /*
    evbid : ident du Buffer
    pdata : adresse du buffer ( donnee par EvbAlloc )

    */
  evbuffer *evb = (evbuffer *)evbid ;
  evb_buf_header *pbuf ;
  error_code ok ;

  if ( (ok = evb_ok( evb )) || (ok = evb_mine( evb )) ) return ok ;
  if ( ok = evb_lock( &evb->exclusion ) ) return ok ;

  pbuf = (evb_buf_header *)((char *)pdata - sizeof( evb_buf_header )) ;
  evb_unlink( pbuf, &evb->buf_head, &evb->buf_tail ) ;
  mem_insert( pbuf, evb ) ;
  evb->nbbuf-- ;
  return evb_unlock( &evb->exclusion ) ;
}

#if EVB_VERSION<=4

static int
in_time( TimeStamp *cur, TimeStamp *min, TimeStamp *max )
{
  /* Compare the time stamp of cur with boundaries.
     return TOO_YOUNG if cur is too young
            IN_TIME if cur in between boundaries
	    TOO_OLD if cur is too old
     Min->second == Max.second
     Min->nano   may be < 0
     Max->nano   may be >MAX_NANO
     */
  if ( cur->second < min->second ) return TOO_OLD ;
  else if ( cur->second > max->second ) return TOO_YOUNG ;
  /* Same second */
  else if ( cur->nano < min->nano ) return TOO_OLD ;
  else if ( cur->nano > max->nano ) return TOO_YOUNG ;
  else return IN_TIME ;
}

static evb_buf_header *EvtFound = NULL ;

static error_code
find_evb( evbuffer *pevb, evb_buf_header *first,
	  void *data, int *size, int *status,
	 TimeStamp *req, int delta )
{
  TimeStamp min, max ;
  evb_buf_header *pf = first ;
  error_code ok ;
  TimeStamp *pdat ;
  int found ;

  /* Calculate min and max */
  min = *req ;
  max = *req ;
  min.nano -= delta ;
  max.nano += delta ;

  pf = pf->next ;
  for( ; pf ; pf = pf->next ) {
    pdat = (void *)((char *)pf + sizeof( evb_buf_header )) ;
    if ( (found = in_time( pdat, &min, &max )) == IN_TIME ) {
      *status = EVENT_FOUND ;
      if ( *size < pf->size ) {
	memcpy( data, pdat, *size ) ;
	*status = EVENT_FOUND_TOO_BIG ;
      }
      else memcpy( data, pdat, pf->size ) ;
      *size = pf->size ;
      EvtFound = pf ;
      evb_unlock( &pevb->exclusion ) ;
      return SUCCESS ;
    }
    else if ( found == TOO_YOUNG ) {
      evb_unlock( &pevb->exclusion ) ;
      *status = EVENT_NOT_FOUND ;
      return SUCCESS ;
    }
  }
  evb_unlock( &pevb->exclusion ) ;
  *status = EVENT_TOO_YOUNG ;
  return SUCCESS ;
}

error_code
EvbFindNext( EvbId evbid, void *data, int *size, int *status,
	 TimeStamp *req, int delta )
{
  evbuffer *pevb = (evbuffer *)evbid ;
  error_code ok ;

  if ( ok = evb_ok( pevb ) ) return ok ;
  if ( ok = evb_lock( &pevb->exclusion ) ) return ok ;
  if ( EvtFound == NULL ) return TOO_YOUNG ;
  return find_evb( pevb, EvtFound, data, size, status,
		   req, delta ) ;
}

error_code
EvbFind( EvbId evbid, void *data, int *size, int *status,
	 TimeStamp *req, int delta )
{
  TimeStamp min, max ;
  evbuffer *pevb = (evbuffer *)evbid ;
  evb_buf_header *pf ;
  error_code ok ;
  TimeStamp *pdat ;
  int found ;

  if ( ok = evb_ok( pevb ) ) return ok ;
  if ( ok = evb_lock( &pevb->exclusion ) ) return ok ;
  /* Calculate min and max */
  min = *req ;
  max = *req ;
  min.nano -= delta ;
  max.nano += delta ;
  /* Check if first evt is younger than requested */
  pf = pevb->buf_head ;
  EvtFound = NULL ;
  pdat = (void *)((char *)pf + sizeof( evb_buf_header )) ;
  if ( ( found = in_time( pdat, &min, &max )) == TOO_YOUNG ) {
    evb_unlock( &pevb->exclusion ) ;
    *status = EVENT_LOST ;
    return SUCCESS ;
  }
  else if ( found == IN_TIME ) {
    *status = EVENT_FOUND ;
    if ( *size < pf->size ) {
      memcpy( data, pdat, *size ) ;
      *status = EVENT_FOUND_TOO_BIG ;
    }
    else memcpy( data, pdat, pf->size ) ;
    *size = pf->size ;
    EvtFound = pf ;
    evb_unlock( &pevb->exclusion ) ;
    return SUCCESS ;
  }
  /* Now check other events */
  return find_evb( pevb, pf, data, size, status,
		   req, delta ) ;
}

error_code
EvbFirst( EvbId evbid, void **pbf, void **data )
{
  evbuffer *pevb = (evbuffer *)evbid ;
  evb_buf_header *pf ;
  error_code ok ;

  if ( ok = evb_ok( pevb ) ) return ok ;
  if ( ok = evb_lock( &pevb->exclusion ) ) return ok ;

  *pbf = pf = pevb->buf_head ;
  *data = (void *)((char *)pf + sizeof( evb_buf_header )) ;
  return SUCCESS ;
}

error_code
EvbNext( EvbId evbid, void **pbf, void **data )
{
  evbuffer *pevb = (evbuffer *)evbid ;
  evb_buf_header *pf ;
  error_code ok ;

  pf = ((evb_buf_header *)*pbf)->next ;
  if ( !pf ) {
    *pbf = NULL ;
    *data = NULL ;
    return SUCCESS ;
  }
  *pbf = pf ;
  *data = (void *)((char *)pf + sizeof( evb_buf_header )) ;
  return SUCCESS ;
}
#else

error_code
EvbSetTag( EvbId evbid, void *pdata, int index, unsigned char tag )
{
  evbuffer *evb = (evbuffer *)evbid ;
  evb_buf_header *pbuf, *nbuf ;
  error_code ok ;

  if ( ok = evb_ok( evb ) ) return ok ;
  if ( ok = evb_lock( &evb->exclusion ) ) return ok ;

  pbuf = (evb_buf_header *)((char *)pdata - sizeof(evb_buf_header) ) ;
  if ( pbuf->buf_ok != EVB_GOOD_BUF ) {
    evb_unlock( &evb->exclusion ) ;
    return EVB_ERR_BAD_BUF_POINTER ;
  }

  pbuf->tag[index] = tag ;
  if ( ok = evb_unlock( &evb->exclusion ) ) return ok ;
  return SUCCESS ;
}

error_code
EvbReady( EvbId evbid, void *pdata, int size, unsigned char *tags )
{
  /* Shrink et rend le buffer accessible */
  evbuffer *evb = (evbuffer *)evbid ;
  evb_buf_header *pbuf, *nbuf ;
  error_code ok ;
  int nsize, i ;

  if ( ok = evb_ok( evb ) ) return ok ;
  if ( ok = evb_lock( &evb->exclusion ) ) return ok ;

  pbuf = (evb_buf_header *)((char *)pdata - sizeof(evb_buf_header) ) ;
  if ( pbuf->buf_ok != EVB_GOOD_BUF ) {
    evb_unlock( &evb->exclusion ) ;
    return EVB_ERR_BAD_BUF_POINTER ;
  }

  if ( pbuf->bsize > (size + sizeof( evb_buf_header ) + MINIMUM_BUF_SIZE) ) {
    /* On peut shrinker le buffer */
    nbuf = (evb_buf_header *)((char *)pbuf + size + sizeof( evb_buf_header) ) ;
    nsize = pbuf->bsize ;
    pbuf->bsize = size + sizeof( evb_buf_header ) ;
    pbuf->size = size ;
    nbuf->bsize = nsize - ( pbuf->bsize ) ;
    mem_insert( nbuf, evb ) ;
    garbage_collect( evb, nbuf, evb->free_tail ) ;
  }
  /* mettre le buffer en queue de la liste des messages */
  evb_tail( pbuf, &evb->buf_head, &evb->buf_tail ) ;
  evb->nbbuf++ ;
  if ( tags ) for( i = 0 ; i < 16 ; pbuf->tag[i] = tags[i], i++ ) ;
  else for( i = 0 ; i < 16 ; pbuf->tag[i] = 0, i++ ) ;
  if ( ok = evb_unlock( &evb->exclusion ) ) return ok ;
  return SUCCESS ;
}

/* 
   EvbFindTag:

   La fonction compare retourne :
   EVB_ERR_NOT_FOUND si pas trouve. Dans ce cas on passe au suivant.
   SUCCESS si trouve. On copie les data au bon endroit, on met a jour
     le tag, et on retourne SUCCESS.
   Autre chose. on ne copie pas les donnees, on ne met pas a jour le tag,
     et on retourne "autre chose".
*/

error_code
EvbFindTag( EvbId evbid, void *pdat, int *size,
	    int (*compare)( void *, unsigned char),
	    int index, unsigned char tag, unsigned char newtag )
{
  evbuffer *pevb = (evbuffer *)evbid ;
  evb_buf_header *pf ;
  error_code ok ;
  unsigned char *data ;
  int found ;
  if ( ok = evb_ok( pevb ) ) return ok ;
  if ( ok = evb_lock( &pevb->exclusion ) ) return ok ;

  pf = pevb->buf_head ;
  for( ; pf ; pf = pf->next )
    if ( pf->tag[index] == tag ) {
      data = (void *)((char *)pf + sizeof( evb_buf_header )) ;
      if ( (found = compare( data, pf->tag[index] )) == EVB_ERR_NOT_FOUND )
	continue ;
      else if ( found == SUCCESS ) {
	if ( *size < pf->size ) memcpy( pdat, data, *size ) ;
	else memcpy( pdat, data, pf->size ) ;
	*size = pf->size ;
	pf->tag[index] = newtag ;
	evb_unlock( &pevb->exclusion ) ;
	return SUCCESS ;
      }
      else {
	  evb_unlock( &pevb->exclusion ) ;
	  return found ;
      }
    }
  evb_unlock( &pevb->exclusion ) ;
  return EVB_ERR_NOT_FOUND ;
}

error_code
EvbFind( EvbId evbid, void *pdat, int *size,
	    int (*compare)( void *, unsigned char) )
{
  evbuffer *pevb = (evbuffer *)evbid ;
  evb_buf_header *pf ;
  error_code ok ;
  unsigned char *data ;
  int found ;

  if ( ok = evb_ok( pevb ) ) return ok ;
  if ( ok = evb_lock( &pevb->exclusion ) ) return ok ;

  pf = pevb->buf_head ;
  for( ; pf ; pf = pf->next ) {
    data = (void *)((char *)pf + sizeof( evb_buf_header )) ;
    if ( (found = compare( data, 0 )) == EVB_ERR_NOT_FOUND )
      continue ;
    else if ( found == SUCCESS ) {
      if ( *size < pf->size ) memcpy( pdat, data, *size ) ;
      else memcpy( pdat, data, pf->size ) ;
      *size = pf->size ;
      evb_unlock( &pevb->exclusion ) ;
      return SUCCESS ;
    }
    else {
      evb_unlock( &pevb->exclusion ) ;
      return found ;
    }
  }
  evb_unlock( &pevb->exclusion ) ;
  return EVB_ERR_NOT_FOUND ;
}

error_code
EvbGetFirstTag( EvbId evbid, void *pdat, int *size, int index,
		unsigned char tag, unsigned char newtag )
{
  evbuffer *pevb = (evbuffer *)evbid ;
  evb_buf_header *pf ;
  error_code ok ;
  unsigned char *data ;

  if ( ok = evb_ok( pevb ) ) return ok ;
  if ( ok = evb_lock( &pevb->exclusion ) ) return ok ;

  pf = pevb->buf_head ;
  for( ; pf ; pf = pf->next ) if ( pf->tag[index] == tag ) {
    data = (void *)((char *)pf + sizeof( evb_buf_header )) ;
    if ( *size < pf->size ) memcpy( pdat, data, *size ) ;
    else memcpy( pdat, data, pf->size ) ;
    *size = pf->size ;
    pf->tag[index] = newtag ;
    ok = SUCCESS ;
  }
  else ok = EVB_ERR_NOT_FOUND ;
  evb_unlock( &pevb->exclusion ) ;
  return ok ;
}

error_code
EvbGetLastTag( EvbId evbid, void *pdat, int *size, int index,
		unsigned char tag, unsigned char newtag )
{
  evbuffer *pevb = (evbuffer *)evbid ;
  evb_buf_header *pf ;
  error_code ok ;
  unsigned char *data ;

  if ( ok = evb_ok( pevb ) ) return ok ;
  if ( ok = evb_lock( &pevb->exclusion ) ) return ok ;

  ok = EVB_ERR_NOT_FOUND ;
  pf = pevb->buf_tail ;
  for( ; pf ; pf = pf->prev ) if ( pf->tag[index] == tag ) {
    data = (void *)((char *)pf + sizeof( evb_buf_header )) ;
    if ( *size < pf->size ) memcpy( pdat, data, *size ) ;
    else memcpy( pdat, data, pf->size ) ;
    *size = pf->size ;
    pf->tag[index] = newtag ;
    ok = SUCCESS ;
  }

  evb_unlock( &pevb->exclusion ) ;
  return ok ;
}

error_code
EvbGetLatestTag( EvbId evbid, void *pdat, int *size, int index,
		unsigned char tag, unsigned char newtag )
{
  /* rend le plus recent tagge a condition qu'il n'y en a pas de plus
     recent deja "newtagge" */
  evbuffer *pevb = (evbuffer *)evbid ;
  evb_buf_header *pf ;
  error_code ok ;
  unsigned char *data ;

  if ( ok = evb_ok( pevb ) ) return ok ;
  if ( ok = evb_lock( &pevb->exclusion ) ) return ok ;

  ok = EVB_ERR_NOT_FOUND ;
  pf = pevb->buf_tail ;
  for( ; pf ; pf = pf->prev )
    if ( pf->tag[index] == tag ) {
      data = (void *)((char *)pf + sizeof( evb_buf_header )) ;
      if ( *size < pf->size ) memcpy( pdat, data, *size ) ;
      else memcpy( pdat, data, pf->size ) ;
      *size = pf->size ;
      pf->tag[index] = newtag ;
      ok = SUCCESS ;
    }
  else if ( pf->tag[index] == newtag ) break ;

  evb_unlock( &pevb->exclusion ) ;
  return ok ;
}

#endif
