#include <stdio.h>

/*************************************************************
This set of functions is intended to handle properly linked
lists of items. The items may be any type of structure, provided
that :
	the 2 first elements of the structure are pointers to NEXt and PREVIOUS
	elements in the list.
	The head and tail of the list are pointeurs to elements of the list, and
	are passed by reference to the functions lkl_after, lkl_before, lkl_unlink.
	Tail and head must be initialized a NULL to define an EMPTY list.
	
Example:

	typedef struct item_st MY_ITEM ;
	struct item_st {
		MY_ITEM *next, *prev ;	<----- MANDATORY !!!!!
		char *name ;
		int status, value ;
		......
	} ;
	MY_ITEM *First_item = NULL, *Last_item = NULL ;
	
	***** Adding new items at the beginning of the list *****

	MY_ITEM *new0, *new1 ;
	new0= malloc( sizeof( MY_ITEM ) ) ;

First_item -------> NULL     <-------- Last_item

	lkl_before( new, First_item, &First_item, &Last_item ) ;
	
First_item ---+      new0         +--- Last_item
              +-> +----------+ <--+
                  |  NULL    |
                  |  NULL    |
                  | .......  |
                  +----------+

	new1 = malloc( sizeof( MY_ITEM ) ) ;
	lkl_before( new1, First_item, &First_item, &Last_item ) ;
	
First_item ----        new1                 new0           ----  Last_item
              |----> +----------+    ---> +----------+  <--|
                |    |  new0    |->--|    |  NULL    |
                |    |  NULL    |      -<-|  new1    |
                |    | .......  |      |  | .......  |
                |    +----------+      |  +----------+
                |                      |
                ------------------------
                
***************************************************************************/

typedef struct s_item GENERIC_ITEM ;

struct s_item {
	GENERIC_ITEM *next, *prev ;
} ;


void *
lkl_tail( GENERIC_ITEM *new, GENERIC_ITEM **first, GENERIC_ITEM **last )
{
  GENERIC_ITEM *old ;
  if ( !(*first) ) {	/* List is empty */
    *first = *last = new ;
    new->next = new->prev = NULL ;
  }
  else {
    old = *last ;
    new->next = old->next ;
    new->prev = old ;
    old->next = new ;
    *last = new ;
  }
  return new ;
}

void *
lkl_after( GENERIC_ITEM *new, GENERIC_ITEM *old,
	   GENERIC_ITEM **first, GENERIC_ITEM **last )
{
  if ( !(*first) ) {	/* List is empty */
    *first = *last = new ;
    new->next = new->prev = NULL ;
  }
  else {
    if ( !old ) return lkl_tail( new, first, last ) ;
    if ( old->next ) old->next->prev = new ;
    else *last = new ;
    new->next = old->next ;
    new->prev = old ;
    old->next = new ;
  }
  return new ;
}

void *
lkl_head( GENERIC_ITEM *new, GENERIC_ITEM **first, GENERIC_ITEM **last )
{
  GENERIC_ITEM *old ;
  if ( !(*first ) ) {	/* List is empty */
    new->next = new->prev = NULL ;
    *first = *last = new ;
  }
  else {
    old = *first ;
    new->next = old ;
    new->prev = NULL ;
    old->prev = new ;
    *first = new ;
  }
  return new ;
}

void *
lkl_before( GENERIC_ITEM *new, GENERIC_ITEM *old,
	    GENERIC_ITEM **first, GENERIC_ITEM **last )
{
  if ( !(*first ) ) {	/* List is empty */
    new->next = new->prev = NULL ;
    *first = *last = new ;
  }
  else {
    if ( !old ) return lkl_head( new, first, last ) ;
    if ( old->prev ) old->prev->next = new ;
    else *first = new ;
    new->next = old ;
    new->prev = old->prev ;
    old->prev = new ;
  }
  return new ;
}


void *
lkl_unlink( GENERIC_ITEM *old, GENERIC_ITEM **first, GENERIC_ITEM **last )
{
  if ( !old || !(*first) ) return NULL ;
  if ( old->next ) old->next->prev = old->prev ;
  else *last = old->prev ;
  if ( old->prev ) old->prev->next = old->next ;
  else *first = old->next ;
  return old ;
}

void *
lkl_find( GENERIC_ITEM *first, GENERIC_ITEM *last, int (*func)() )
{
  GENERIC_ITEM *cur = first ;
  for ( cur = first ; cur ; cur = cur->next )
    if ( func( cur ) == 0 ) return cur ;
  return NULL ;
}
