/******************************************************************

  PostLib

  Messagerie inter-process.
  Un process cree une boite postale ( PostCreate ). Il sera le seul a pouvoir
  lire les messages. Lors de la creation il decide de la maniere dont il sera
  informe que des messages sont disponibles dans la boite.
  Des process se link a la boite postale ( PostLink ). Il ne pourrontqu' y
  envoyer des messages, mais n'auront pas le droit de les lire.
  La librairie fait 2 choses a la fois :
    1. Gerer la memeoire dans le data module "a la maniere de" malloc avec
       les fonctions PostAlloc et PostCancel.
    2. Gerer la messagerie proprement dite avec les autres fonctions
  Pour minimiser les recopies, l'envoi d'un message se passe en 3 temps:
    1. Reserver de la place pour y creer le message - PostAlloc
    2. Creer le message dans la zone allouee.
    3. Envoyer le message - PostSend
  De la meme maniere la lecture du message se fait en 3 temps:
    1. Obtenir l'adresse du mesage - PostGet ( le message est enleve de la 
       liste des messages en attente ).
    2. Traiter le message 
    3. Remettre la zone dans la liste des zones libres. - PostCancel

  La boite postale se trouve dans un DataModule.
  On a la-dedans
    La liste des zones libres.
    La liste ( plus ancien d'abord ) des messages delivres.
  L'exclusion mutuelle ( reservation et cancel de messages ) se fait via un
  semaphore.

  *************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <semaphore.h>
#include <signal.h>
#include <events.h>
#include <process.h>
#include <errno.h>

#include "postlib.h"
#include "postdefs.h"

#define POST_GOOD_MSG 0xBABADEDE

static char Version[64] ;
static MailboxId Mbxid = 0 ;

static char *Post_error_string[] = {
  "No memory to create the mailbox",
  "Mailbox Unknown or Inactive",
  "Mailbox Already Exists",
  "No Message available in the Mailbox",
  "Not Enough Space in the Mailbox",
  "The Mailbox is Owned by an Other Process",
  "Bad Notify Mode",
  "Bad Message Pointer",
  "Unknown Error, Contact L.Guglielmi",
  NULL
} ;

void postcheck( MailboxId mbxid, char *where )
{
#if defined(POST_DEBUG)
  mailbox *pmbx = (mailbox *)mbxid ;
  post_msg_header *pmsg ;

  if ( mbxid != Mbxid ) {
    printf( "\n>>>>>>>>> %x au lieu de %x\n", mbxid, Mbxid ) ;
    fflush( stdout ) ;
    exit( 1 ) ;
  }
  if ( pmbx->msg_head == (post_msg_header *)0x7c200026 ) {
    printf( "\n****Head**** %s, nbmess : %d (%d, %d) \n",
	    where, pmbx->nbmess, pmbx->totmsgin, pmbx->totmsgout ) ;
    fflush( stdout ) ;
    exit( 1 ) ;
  }
  else if ( pmbx->msg_head == (post_msg_header *)0x4 ) {
    printf( "\n====Head==== %s, nbmess : %d (%d, %d) \n",
	    where, pmbx->nbmess, pmbx->totmsgin, pmbx->totmsgout ) ;
    fflush( stdout ) ;
    exit( 1 ) ;
  }
  if ( pmbx->msg_head != (post_msg_header *)0 &&
       pmbx->msg_tail == (post_msg_header *)0 ) {
    printf( "\n****Tail**** %s, nbmess : %d (%d, %d) \n",
	    where, pmbx->nbmess, pmbx->totmsgin, pmbx->totmsgout ) ;
    fflush( stdout ) ;
    exit( 1 ) ;
  }
  for( pmsg = pmbx->msg_head ; pmsg ; pmsg = pmsg->next )
    if ( pmsg->next == (post_msg_header *)0x7c200026 ) {
      printf( "\n+++++++ %s, pmsg->next Bad\n", where ) ;
      exit( 1 ) ;
    }
    else if ( pmsg->next == (post_msg_header *)0x4 ) {
      printf( "\n------- %s, pmsg->next Bad\n", where ) ;
      exit( 1 ) ;
    }
#endif
}

static error_code
post_ok( mailbox *mbx )
{
  if ( !mbx || mbx->synch != POST_SYNCH || mbx->active != POST_ACTIVE )
    return POST_ERR_UNKNOWN ;
  return E_SUCCESS ;
}

static error_code
post_mine( mailbox *mbx )
{
  if ( mbx->owner != my_pid() ) return POST_ERR_NOTYOURS ;
  return E_SUCCESS ;
}

static error_code
post_create( char *mname, int size, void **head, mh_com **addr )
{
  unsigned short attrev = 0x8000, /* Sharable, Sticky Bit */
    typlang = 0x400; /* Type "data module" */
  unsigned long perm = 0x333; /* RW pour tout le monde */
  error_code ok ;

  ok = _os_datmod(
		  mname,
		  size,
		  &attrev,
		  &typlang,
		  perm,
		  head,
		  addr) ;
    /* the data module has just been created */

    return ok ;
}

static error_code
post_link(  char *mname, void **head, mh_com **addr )
{
  unsigned short attrev = 0x8000, /* Sharable, Sticky Bit */
    typlang = 0x400; /* Type "data module" */
  unsigned long perm = 0x333; /* RW pour tout le monde */
  error_code ok ;

  ok = _os_link(&mname,
		addr,
		head,
		&typlang,
		&attrev ) ;
  /* the data-module already exists */
  return ok ;
}

static error_code
post_finish( MailboxId *mbxid, mh_com *addr )
{
  if ( *mbxid ) _os_unlink ( addr ) ;
  *mbxid = 0 ;
  return 0 ;
}

#if POST_OFFICE_VERSION==99

static error_code
post_lock( event_id *evid )
{
  int value ;
  signal_code dummy ;

  return _os_ev_wait( *evid, &value, &dummy, 0, 0 ) ;
}

static error_code
post_unlock( event_id *evid )
{
  int value ;

  return _os_ev_signal( *evid, &value, 0 ) ;
}

#else

static error_code
post_lock( semaphore *sem )
{
  while( _os_sema_p( sem ) != E_SUCCESS ) ;
  return E_SUCCESS ;
}

static error_code
post_unlock( semaphore *sem )
{
  while( _os_sema_v( sem ) != E_SUCCESS ) ;
  return E_SUCCESS ;
}
#endif

static error_code
notify( mailbox *mbx )
{
  /* Notify receiver */
  int dummy ;
  error_code ok ;

  if ( mbx->notify.mode == POST_NTFY_NONE ) return E_SUCCESS ;
  switch ( mbx->notify.mode ) {
  case POST_NTFY_SETOR_EVENT : return
				 _os_ev_setor(
					      (event_id)mbx->notify.param1,
					      &dummy,
					      (u_int32)mbx->notify.param2,
					      0 ) ;
  case POST_NTFY_SETAND_EVENT : return 
				  _os_ev_setand(
						(event_id)mbx->notify.param1,
						&dummy,
						(u_int32)mbx->notify.param2,
						0 ) ;
  case POST_NTFY_SIGNAL : ok = _os_send(
					mbx->owner,
					(signal_code)mbx->notify.param1 ) ;
    if ( ok == E_SIGNAL ) ok = E_SUCCESS ;
    return ok ;
  case POST_NTFY_EVENT  : return _os_ev_signal(
					       (event_id)mbx->notify.param1,
					       &dummy,
					       0 ) ; break ;
  case POST_NTFY_FLAG   : mbx->notify.param1 += 1 ; return E_SUCCESS ;
  default : return POST_ERR_BAD_MODE ;
  }
}

/***************************************************
  Gestion des listes chainees

  post_unlink
  post_tail
  post_head
  post_before
  **************************************************/

static void
post_tail( post_msg_header *new, post_msg_header **first,
	   post_msg_header **last )
{
  post_msg_header *old ;


  if ( !(*first) ) {
    *first = *last = new ;
    new->next = new->prev = NULL ;
  }
  else {
    old = *last ;
    new->next = old->next ;
    new->prev = old ;
    old->next = new ;
    *last = new ;
  }
}

#if 0
void 
post_head( post_msg_header *new,
	   post_msg_header **first,
	   post_msg_header **last )
{
  post_msg_header *old ;
  if ( !(*first ) ) {	/* List is empty */
    new->next = new->prev = NULL ;
    *first = *last = new ;
  }
  else {
    old = *first ;
    new->next = old ;
    new->prev = NULL ;
    old->prev = new ;
    *first = new ;
  }
}
#endif

void
post_before( post_msg_header *new, 
	     post_msg_header *old,
	     post_msg_header **first,
	     post_msg_header **last )
{
    new->prev = old->prev ;
    old->prev = new ;
    if ( new->prev ) new->prev->next = new ;
    else *first = new ;
    new->next = old ;
}

void
post_unlink( post_msg_header *old,
	     post_msg_header **first,
	     post_msg_header **last )
{
  if ( old->next ) old->next->prev = old->prev ;
  else *last = old->prev ;
  if ( old->prev ) old->prev->next = old->next ;
  else *first = old->next ;
}

/*****************************************************
  Gestion de la memoire
  ***************************************************/

static post_msg_header *
mem_extract( int size, mailbox *mbx )
{
  /* reherche parmi les zones libres une zone de taille suffisante
     et mettre a jour la liste.
     Il faudra y ajouter la glue pour avoir un multiple de 4 bytes */
  post_msg_header *pmsg = mbx->free_head, *nmsg ;
  int bsize, msize ;

  bsize = size + sizeof( post_msg_header ) ;
#if POST_OFFICE_VERSION>2
  if ( (bsize & 0x3 ) != 0 ) bsize = ( bsize & 0xFFFFFFFC ) + 4 ;
#endif
  msize = bsize + sizeof( post_msg_header ) + MINIMUM_MSG_SIZE ;
  for( ; pmsg ; pmsg = pmsg->next ) {
    if ( pmsg->bsize >= bsize ) {
      /* trouve un bloc assez grand, y a pluka */
      /* si taille du bloc trouve - taille du nouveau bloc < minimum alors
	     prendre tout le bloc.
	 sinon creer un bloc de la taille adequate */
      if ( pmsg->bsize >= msize ) {
	pmsg->bsize = pmsg->bsize - bsize ;
	nmsg = (post_msg_header *)((int)pmsg + pmsg->bsize ) ;
	nmsg->bsize = bsize ;
      }
      else {
	nmsg = pmsg ;
	post_unlink( pmsg, &mbx->free_head, &mbx->free_tail ) ;
      }
      nmsg->prev = nmsg->next = NULL ;
      nmsg->msg_ok = POST_GOOD_MSG ;
      nmsg->size = size ;
      return nmsg ;
    }
  }
  return NULL ;
}

void static
garbage_collect( mailbox *mbx )
{
  post_msg_header *p ;
  int bsize, success = 0 ;

  /* concatener les blocks libres consecutifs */
  for( p = mbx->free_head ; p != mbx->free_tail ; )
    if ( (unsigned int)p->next == ((unsigned int)p + p->bsize ) ) {
      success = 1;
      bsize = p->next->bsize ;
      if ( p->next == mbx->free_tail ) {
	mbx->free_tail = p ;
	p->next = NULL ;
	p->bsize += bsize ;
	break ;
      }
      else post_unlink( p->next, &mbx->free_head, &mbx->free_tail ) ;
      p->bsize += bsize ;
    }
    else p = p->next ;
  mbx->gattempt++ ;
  mbx->gsuccess += success ;
}

static void
mem_insert( post_msg_header *pmsg, mailbox *mbx )
{
  post_msg_header *p ;

  /* Il faudrait inserer le block dans l'ordre des adresses croissantes
     ce qui permettrait de faciliter la garbage collection */
  pmsg->size = 0 ;
  pmsg->msg_ok = 0 ;
  /* rechercher l'endroit ou inserer ce bloc */
  for( p = mbx->free_head ; p != mbx->free_tail ; p = p->next )
    if ( p > pmsg ) {
      /* C'est ici qu'il faut l'inserer */
      post_before( pmsg, p, &mbx->free_head, &mbx->free_tail ) ;
      return ;
    }
  if ( mbx->free_tail < pmsg ) 
    post_tail( pmsg, &mbx->free_head, &mbx->free_tail ) ;
  else
    post_before( pmsg, mbx->free_tail, &mbx->free_head, &mbx->free_tail ) ;
}

/**********************************************************
  Les vraies fonctions
  *********************************************************/

int
PostVersionId()
{
  return POST_OFFICE_VERSION ;
}

char *
PostVersion()
{
  sprintf( Version, "Post Office Library V%d ( %s )", POST_OFFICE_VERSION, __DATE__ ) ;
  return Version ;
}

char *
PostError( error_code ok )
{
  if ( ok < POST_ERR_NO_MEMORY || ok >= POST_UNKNOWN_ERROR )
    ok = POST_UNKNOWN_ERROR ;
  return Post_error_string[ok - POST_ERR_NO_MEMORY] ;
}

/*void
PostDump( MailboxId mbxid, FILE *f )
{
  mailbox *pmbx = (mailbox *)mbxid ;
  post_msg_header *pf ;
  error_code ok ;
  int i ;

  if ( ok = post_ok( pmbx ) )
    fprintf( f, "%s\n", PostError( POST_ERR_UNKNOWN ) ) ;

  fprintf( f, ">>>>>>>> Mailbox Dump\n" ) ;
  fprintf( f, "Mailbox Address : %x\n", pmbx ) ;
  fprintf( f, " Version  = %d\n", pmbx->version ) ;
  fprintf( f, " Synch    = %x\n", pmbx->synch ) ;
  fprintf( f, " Activ    = %x\n", pmbx->active ) ;
  fprintf( f, " Size     = %d\n", pmbx->size ) ;
  fprintf( f, " Allocated = %d, Deallocated = %d\n", pmbx->totalloc,
	   pmbx->totdealloc ) ;
  fprintf( f, " Nbmess   = %d ( in %d, out %d )\n",
	   pmbx->nbmess, pmbx->totmsgin, pmbx->totmsgout ) ;
  fprintf( f, " Owner    = %d\n", pmbx->owner ) ;
  fprintf( f, " Semaphore :\n" ) ;
  fprintf( f, 
	   "    value  : %d, lock : %d\n", pmbx->exclusion.s_value,
	   pmbx->exclusion.s_lock ) ;
  fprintf( f,"    s_qnext : %x, s_qprev : %x\n", pmbx->exclusion.s_qnext,
	   pmbx->exclusion.s_qprev ) ;
  fprintf( f,"    length : %d, owner : %d\n", pmbx->exclusion.s_length,
	   pmbx->exclusion.s_owner ) ;
  fprintf( f, " Garbage collections %d ( %d successful )\n",
	   pmbx->gattempt, pmbx->gsuccess ) ;
  fprintf( f, "+++++ Free Blocks : Head = %x, Tail = %x\n",
	   pmbx->free_head, pmbx->free_tail ) ;
  for( i=0, pf = pmbx->free_head ; pf ; pf = pf->next, i++ )
    fprintf( f, "  Block Size = %d, Addr = %x to %x\n", 
	     pf->bsize, pf, (unsigned int)pf + pf->bsize ) ;
  fprintf( f, "+++++ Msg Blocks  : Head = %x, Tail = %x\n",
	   pmbx->msg_head, pmbx->msg_tail ) ;
  for( i=0, pf = pmbx->msg_head ; pf ; pf = pf->next, i++ )
    fprintf( f, "  Block Size = %d, Msg Size = %d, Addr = %x\n",
	     pf->bsize, pf->size, pf ) ;

}*/

error_code
PostSetMode( MailboxId mbxid, post_notify *mode )
{
  mailbox *mbx = (mailbox *)mbxid ;
  error_code ok ;

  if ( ok = post_mine( mbx ) ) return ok ;
  if ( ok = post_lock( &mbx->exclusion ) ) return ok ;
  mbx->notify = *mode ;
  return post_unlock( &mbx->exclusion ) ;
}

error_code
PostCreate( MailboxId *mbxid, char *name, int size, post_notify *mode )
{
  /* **mbx : adress de l'identificateur de la boite postale.
     *name : pointeur sur le nom de la boite postale.
     size  : taille de la boite postale.
     *mode : pointeur sur la strcuture decrivant le mode de notification.

     Code de retour :
       POST_ERR_ALREADY_EXIST	: la boite existe deja.
       POST_ERR_NO_MEMORY	: pas de place en memoire pour creer
                                  le data module

				  */
  mailbox *pmbx ;
  mh_com *addr ;
  error_code ok ;

  if ( ok = post_create( name, size+sizeof( mailbox ), (void **)&pmbx, &addr ) )
    return ok ;

  /* Mailbox created, now initialize it */
  pmbx->version = POST_OFFICE_VERSION ;
  pmbx->owner = my_pid() ;
  pmbx->notify = *mode ;
  pmbx->nbmess = 0 ;
  pmbx->size = size ;
  pmbx->addr = addr ;
#if POST_OFFICE_VERSION==99
  /* Initialise the OS9 Event */
  ok = _os_ev_creat( 1, -1, 0x777, &pmbx->exclusion,
		     name, 0, 0 ) ;
#else
  /* Initialise the semaphore */
  if( ok = _os_sema_init( &pmbx->exclusion ) ) {
    post_finish( mbxid, pmbx->addr ) ;
    return ok ;
  }
#endif
  /* Initialize pointers */
  pmbx->memory_arena = ( void *)(pmbx+1) ;
  pmbx->msg_head = pmbx->msg_tail = NULL ;
  pmbx->free_head = (post_msg_header *)(pmbx + 1) ;
  pmbx->free_tail = pmbx->free_head ;
  pmbx->free_head->prev = pmbx->free_head->next = NULL ;
  pmbx->free_head->bsize = (int)pmbx->size ;
  pmbx->synch = POST_SYNCH ;
  pmbx->active = POST_ACTIVE ;
  *mbxid = (MailboxId)pmbx ;
  Mbxid = (MailboxId)pmbx ;
  return E_SUCCESS ;
}

error_code
PostClose( MailboxId *mbxid )
{
  mailbox *pmbx = *(mailbox **)mbxid ;
  error_code ok ;

  if ( ok = post_ok( pmbx ) ) return ok ;
  if ( post_mine( pmbx ) == E_SUCCESS ) pmbx->active = 0 ;
  post_finish( mbxid, pmbx->addr ) ;
  *mbxid = 0 ;
  return E_SUCCESS ;
}

error_code
PostCleanUp( MailboxId mbxid )
{
  mailbox *mbx = (mailbox *)mbxid ;
  error_code ok ;

  if ( ok = post_ok( mbx ) ) return ok ;
  postcheck( mbxid, "Avant Cleanup" ) ;
  if ( ok = post_lock( &mbx->exclusion ) ) return ok ;
  garbage_collect( mbx ) ;
  postcheck( mbxid, "Apres Cleanup" ) ;
  return ok = post_unlock( &mbx->exclusion ) ;
}

error_code
PostLink( MailboxId *mbxid, char *name )
{
  /* **mbx : adresse de l'identificateur de la boite postale rendu par la
             fonction.
     *name : pointeur sur le nom.

     Code de retour:
       POST_ERR_UNKNOWN	: boite postale inexistante.

       */
  mailbox *pmbx ;
  mh_com *addr ;

  /* if ( CANT link to data module ) then return error
     else if ( NOT an active mailbox ) then
       unlink data module
       return error
     else return OK
     endif
     */
  if ( post_link( name, (void **)&pmbx, &addr ) )
    return POST_ERR_UNKNOWN ;
  else {
    if ( pmbx->synch != POST_SYNCH || pmbx->active != POST_ACTIVE ) {
      post_finish( mbxid, addr ) ;
      return POST_ERR_UNKNOWN ;
    }
  }
  *mbxid = (MailboxId)pmbx ;
  Mbxid = (MailboxId)pmbx ;
  return E_SUCCESS ;
}

error_code
PostDealloc( MailboxId mbxid, void *pdata )
{
  /*
    mbxid : ident de la mailbox
    pdata : adresse du buffer ( donnee par PostAlloc )

    */
  mailbox *mbx = (mailbox *)mbxid ;
  post_msg_header *pmsg ;
  error_code ok ;

  if ( (ok = post_ok( mbx )) ) return ok ;
  if ( ok = post_lock( &mbx->exclusion ) ) return ok ;

  pmsg = (post_msg_header *)((int)pdata - sizeof( post_msg_header )) ;
  mem_insert( pmsg, mbx ) ;
  mbx->totdealloc++ ;
  return post_unlock( &mbx->exclusion ) ;
}

error_code
PostAlloc( MailboxId mbxid, void **data, int size )
{
  /*
     *mbx ; identificateur de la boite postale.
     **data : adresse du buffer donne par la fonction.
     size   ; taille du buffer demande.

     Code de retour:
       POST_ERR_UNKNOWN	: la boite postale n'existe pas ( ou plus !)
       POST_ERR_FULL	: la boite est pleine.

       */
  error_code ok ;
  post_msg_header *pdat ;
  mailbox *mbx = (mailbox *)mbxid ;

  if ( ok = post_ok( mbx ) ) return ok ;
  postcheck( mbxid, "PostAlloc debut" ) ;
  if ( ok = post_lock( &mbx->exclusion ) ) return ok ;;
  if ( mbx->synch != POST_SYNCH || mbx->active != POST_ACTIVE ) {
    post_unlock( &mbx->exclusion ) ;
    return POST_ERR_UNKNOWN ;
  }
  /* Chercher parmi les blocs libres s'il y a de la place */
  pdat = mem_extract( size, mbx ) ;
  postcheck( mbxid, "PostAlloc apres mem_extract" ) ;
  if ( !pdat ) {
#if POST_OFFICE_VERSION>0
    garbage_collect( mbx ) ;
    postcheck( mbxid, "PostAlloc apres garbage_collect" ) ;
    if ( !(pdat = mem_extract( size, mbx )) ) {
      post_unlock( &mbx->exclusion ) ;
      return POST_ERR_FULL ;
    }
#else
    post_unlock( &mbx->exclusion ) ;
    return POST_ERR_FULL ;
#endif
  }
  mbx->totalloc++ ;
  *data = (void *)((int)pdat + sizeof( post_msg_header )) ;
  postcheck( mbxid, "PostAlloc fin" ) ;
  post_unlock( &mbx->exclusion ) ;
  return E_SUCCESS ;
}

error_code
PostSend( MailboxId mbxid, void *data )
{
  /*
     mbxid ; identificateur de la boite postale.
     *data : adresse du buffer donne par la fonction.

     Code de retour:
       POST_ERR_UNKNOWN	: la boite postale n'existe pas ( ou plus !)
       */
  post_msg_header *pmsg ;
  mailbox *mbx = (mailbox *)mbxid ;
  error_code ok ;

  if ( ok = post_ok( mbx ) ) return ok ;
  postcheck( mbxid, "PostSend debut" ) ;
  if ( ok = post_lock( &mbx->exclusion ) ) return ok ;
  /* mettre le buffer en queue de la liste des messages */
  pmsg = (post_msg_header *)((int )data - sizeof(post_msg_header) ) ;
  if ( pmsg->msg_ok != POST_GOOD_MSG ) {
    post_unlock( &mbx->exclusion ) ;
    return POST_ERR_BAD_MSG_POINTER ;
  }
  post_tail( pmsg, &mbx->msg_head, &mbx->msg_tail ) ;
  mbx->totmsgin++ ;
  mbx->nbmess++ ;
  postcheck( mbxid, "PostSend avant notify" ) ;
  ok = notify( mbx ) ;
  postcheck( mbxid, "PostSend apres  notify" ) ;
  post_unlock( &mbx->exclusion ) ;
  return ok ;
}

error_code
PostSendShrink( MailboxId mbxid, void *pdata, int size )
{
  /* Comme PostSend, mais shrink le buffer a la taille "size" */
  mailbox *mbx = (mailbox *)mbxid ;
  post_msg_header *pmsg, *nmsg ;
  error_code ok ;
  int nsize ;

  if ( ok = post_ok( mbx ) ) return ok ;
  postcheck( mbxid, "PostSendShrink debut" ) ;
  if ( ok = post_lock( &mbx->exclusion ) ) return ok ;

  pmsg = (post_msg_header *)((int )pdata - sizeof(post_msg_header) ) ;
  if ( pmsg->msg_ok != POST_GOOD_MSG ) {
    post_unlock( &mbx->exclusion ) ;
    return POST_ERR_BAD_MSG_POINTER ;
  }

#if POST_OFFICE_VERSION<=2
  if ( pmsg->bsize > (size + sizeof( post_msg_header ) + MINIMUM_MSG_SIZE) ) {
    /* On peut shrinker le block et le message */
    nmsg = (post_msg_header *)((int)pmsg + size + sizeof( post_msg_header) ) ;
    nsize = pmsg->bsize ;
    pmsg->bsize = size + sizeof( post_msg_header ) ;
    pmsg->size = size ;
    nmsg->bsize = nsize - ( pmsg->bsize ) ;
    mem_insert( nmsg, mbx ) ;
  }
#else
  if ( pmsg->bsize > (size + sizeof( post_msg_header ) + MINIMUM_MSG_SIZE) ) {
    /* On peut shrinker le block et le message */
    int oldbsize, newbsize ;

    oldbsize = pmsg->bsize ;
    newbsize = size + sizeof( post_msg_header ) ;
    if ( (newbsize & 0x3 ) != 0 ) newbsize = ( newbsize & 0xFFFFFFFC ) + 4 ;
    if ( (oldbsize - newbsize) <= 0 ) pmsg->size = size ;
    else {
      pmsg->bsize = newbsize ;
      pmsg->size = size ;
      nmsg = (post_msg_header *)((int)pmsg + newbsize ) ;
      nmsg->bsize = oldbsize - newbsize ;
      postcheck( mbxid, "PostSendShrink avant mem_insert" ) ;
      mem_insert( nmsg, mbx ) ;
    }
  }
  else {
    /* On shrink seulement le message, le block ne change pas */
    pmsg->size = size ;
  }
#endif
  postcheck( mbxid, "PostSendShrink avant post_tail" ) ;
  /* mettre le buffer en queue de la liste des messages */
  post_tail( pmsg, &mbx->msg_head, &mbx->msg_tail ) ;
  mbx->nbmess++ ;
  mbx->totmsgin++ ;
  if ( ok = post_unlock( &mbx->exclusion ) ) return ok ;
  postcheck( mbxid, "PostSendShrink avant notify" ) ;
  return notify( mbx ) ;
}

error_code
PostPut( MailboxId mbxid, void *data, int size )
{
  /* Fait d'un seul coup : PostAlloc, copy, PostSend */
  void *pdata ;
  error_code ok ;

  if ( ok = PostAlloc( mbxid, (void **)&pdata, size ) ) return ok ;
  memcpy( pdata, data, size ) ;
  return PostSend( mbxid, pdata ) ;
}

error_code
PostReceive( MailboxId mbxid, void **data, int *size )
{
  /*
     *mbx ; identificateur de la boite postale.
     **data : adresse du message donne par la fonction.
     *size   ; taille du message

     Code de retour:
       POST_ERR_UNKNOWN	: la boite postale n'existe pas ( ou plus !)
       POST_ERR_EMPTY	: pas de message dans la boite
       POST_ERR_NOT_YOURS: la boite n'appartient pas a ce process.

       */
  mailbox *mbx = (mailbox *)mbxid ;
  post_msg_header *pmsg ;
  error_code ok ;

  if ( (ok = post_ok( mbx )) || (ok = post_mine( mbx )) ) return ok ;
  postcheck( mbxid, "PostReceive debut" ) ;
  if ( ok = post_lock( &mbx->exclusion ) ) return ok ;
  if ( !mbx->nbmess ) {
    post_unlock( &mbx->exclusion ) ;
    return POST_ERR_EMPTY ;
  }
  postcheck( mbxid, "PostReceive avant unlink" ) ;
  /* Recuperer le message qui se trouve en tete de liste */
  pmsg = mbx->msg_head ;
  *data = (void *)((int)pmsg + sizeof( post_msg_header ) ) ;
  *size = pmsg->size ;
#if POST_OFFICE_VERSION<4
  /* Unlinker ce buffer */
  mbx->msg_head = pmsg->next ;
  postcheck( mbxid, "PostReceive unlink" ) ;
  if ( mbx->msg_head ) mbx->msg_head->prev = mbx->msg_head ;
#else
  post_unlink( pmsg, &mbx->msg_head, &mbx->msg_tail ) ;
  postcheck( mbxid, "PostReceive apres unlink" ) ;
#endif
  mbx->nbmess-- ;
  mbx->totmsgout++ ;
  postcheck( mbxid, "PostReceive fin" ) ;

  return post_unlock( &mbx->exclusion ) ;
}

error_code
PostCancel( MailboxId mbxid, void *pdata )
{
  /*
    mbxid : ident de la mailbox
    pdata : adresse du buffer ( donnee par PostReceive )

    */
  mailbox *mbx = (mailbox *)mbxid ;
  post_msg_header *pmsg ;
  error_code ok ;

  if ( (ok = post_ok( mbx )) || (ok = post_mine( mbx )) ) return ok ;
  postcheck( mbxid, "PostCancel debut" ) ;
  if ( ok = post_lock( &mbx->exclusion ) ) return ok ;

  pmsg = (post_msg_header *)((int)pdata - sizeof( post_msg_header )) ;
#if 0
  post_unlink( pmsg, &mbx->msg_head, &mbx->msg_tail ) ;
  postcheck( mbxid, "PostCancel apres unlink" ) ;
#endif
  mem_insert( pmsg, mbx ) ;
  postcheck( mbxid, "PostCancel fin" ) ;
  return post_unlock( &mbx->exclusion ) ;
}

error_code
PostGet( MailboxId mbxid, void *data, int *size )
{
  /* Fait d'un seul coup PostReceive, copie dans data, PostCancel */
  error_code ok ;
  void *pdata ;
  int psize ;

  if ( ok = PostReceive( mbxid, (void **)&pdata, &psize ) ) return ok ;

  if ( *size < psize ) memcpy( data, pdata, *size ) ;
  else {
    memcpy( data, pdata, psize ) ;
    *size = psize ;
  }
#if POST_OFFICE_VERSION>1
  return PostCancel( mbxid, pdata ) ;
#else
  return PostCancel( mbxid ) ;
#endif
}
