/*
  Librairie pour gestion des raw buffers par Auger.

  Utilisee par Read_out et Trigger2
    ==> 1 producteur, 1 consomateur obligatoire, taille fixe.
    Ca simplifie les choses.

  2 liste chainees de buffers : les free et les used. La gestion de la liste
    se fait via la librairie "listelib"
  Un buffer dans la liste "free" peut etre NOT_USED, IN_USE ou USED.
  La tache Readout ouvre un buffer ( RAW_open ) ce qui le rend NOT_USED.
  La tache Trigger2 s'attache a un buffer ( RAW_use ) ce qui le
    rend IN_USE.
  La tache Trigger2 rend le buffer ( RAW_unuse_xx ) ce qui le rend USED.

  Un buffer peut etre egalement utilise par un tache de spying.
  La tache de spying s'attache a un buffer ( RAW_spy_xxx ) ce qui le rend,
    en plus de son etat ci-dessus, IN_SPY.
  La tache de spying rend le buffer, ce qui le rend SPYED.

  Pour qu'un buffer soit completement rendu a la liste des free buffers,
  il faut qu'il soit NOT_USED et (SPYED ou NOT_SPYED).
  Le data module contient a la fois les buffers et la table des headers.

     +-----------+-----------+--------------------------------------
     |           |  |  |  |  |              |
     | Header    |  |  |  |  |     Buffers  |
     |           |  |  |  |  |              |
     +-----------+-----------+--------------------------------------
                 ^           ^
  header->list --+           |
         Header->buffers ----+
  */
#include <stdio.h>
#include <process.h>
#include <module.h>
#include <semaphore.h>
#include <types.h>

#define RAW_BUFF_MAIN

#include "auger_err.h"
#include "raw_defs.h"


static process_id
my_pid()
{
  u_int16 dum16 ;
  int dum32 ;
  process_id uid ;
  _os_id( &uid, &dum16, &dum16, &dum32, &dum16, &dum16 ) ;
  return uid ;
}

static void
init_buffers()
{
  unsigned int pb = (unsigned int)RAW_header->buffers ;
  RAW_IDENT *pid = RAW_header->liste, *free_h = NULL, *free_t = NULL ;
  int i = RAW_header->nbuffs ;
  for( ; i ; i--, pid++, pb += RAW_header->bsize ) {
    pid->buffer = (void *)pb  ;
    lkl_head( pid, (void **)&free_h, (void **)&free_t ) ;
  }
  RAW_header->free_head = free_h ;
  RAW_header->free_tail = free_t ;
  RAW_header->used_head = NULL ;
  RAW_header->used_tail = NULL ;
}

static void
init_data()
{
  RAW_header->nfree = RAW_header->nbuffs ;
  RAW_header->nused = 0 ;
  RAW_header->count = 0 ;

  RAW_header->waiting = 0 ;
  RAW_header->wakeup = 0 ;

  RAW_header->spy_waiting = 0 ;
  RAW_header->spy_wakeup = 0 ;

  /* initialiser le semaphore */
  _os_sema_init( RAW_sema ) ;
  init_buffers() ;
}

static error_code
create_buff( char *nom, int size, RAW_BUFF_HEADER **buff, mh_com **mhead )
{
  RAW_BUFF_HEADER *pbuf ;
  u_int16 attr = mkattrevs(MA_REENT | MA_GHOST, 1 ),
    typla = mktypelang( MT_DATA, ML_ANY )  ;
  u_int32 perm = MP_OWNER_READ | MP_OWNER_WRITE |
    MP_GROUP_READ | MP_GROUP_WRITE | MP_WORLD_READ | MP_WORLD_WRITE ;
  error_code error ;

  error = _os_datmod(
		     nom,
		     size,
		     &attr,
		     &typla,
		     perm,
		     (void **)&pbuf,
		     (mh_data **)mhead ) ;
  if ( error == RAW_SUCCESS ) *buff = (RAW_BUFF_HEADER *)pbuf ;
  return error ;
}

static error_code
link_buff( char *name, RAW_BUFF_HEADER **buff, mh_com **mhead )
{
  /* Faire un _os_link */
  RAW_BUFF_HEADER *pbuf ;
  char **nom = &name ;
  u_int16 attr = mkattrevs( MA_REENT | MA_GHOST, 1 ),
    typla = mktypelang( MT_DATA, ML_ANY )  ;
  error_code error ;

  error = _os_link(
		   nom,
		   (mh_com **)mhead,
		   (void **)&pbuf,
		   &typla,
		   &attr ) ;
  if ( error == RAW_SUCCESS ) *buff = pbuf ;
  return error ;
}

error_code
RAW_init( char *name, int nb, int bsize, int *status )
{
  /* Initialisation du data module */
  /*
    si on peut creer le module alors
      Initialiser T1_heads, T1_buffs, NT1Buffs.
      Creer un event T1Evt avec le nom du module.
      Mettre a jour les donnees dans le module.
    sinon si on peut se linker dessus alors
      Initialiser T1_heads et T1_buffs.
      Maj de NT1Buffs et T1Evt d'apres ce qui se trouve dans le module.
    sinon grave erreur !
    */
  unsigned int lsize ;
  error_code error ;
  /* Full size of data module = size of the header +
                                nb-of-buffers * size_of_a_buffer +
				nb_of_buffers * size_of_an_ident
  */
  lsize = (bsize*nb) + (sizeof( RAW_IDENT )*nb) + sizeof( RAW_BUFF_HEADER ) ;
  if ( (error = link_buff( name, &RAW_header, &RAW_addr )) == RAW_SUCCESS ) {
    /* Data Module already existing */
    RAW_sema = &RAW_header->raw_sema ;
    *status = RAW_ALREADY_EXISTED ;
  }
  else if ( (error = create_buff( name, lsize, &RAW_header, &RAW_addr )) ==
       RAW_SUCCESS ) {
    /* Successfully created Data Module */
    RAW_header->nbuffs = nb ;
    RAW_header->bsize = bsize ;
#if defined(OLD_ONE)
    RAW_header->buffers = (void *)(RAW_header + 1 ) ;
    RAW_header->liste = (RAW_IDENT *)((unsigned int)RAW_header->buffers +
				    RAW_header->nbuffs ) ;
#else
    RAW_header->liste = (RAW_IDENT *)(RAW_header + 1 ) ;
    RAW_header->buffers = (void *)(RAW_header->liste + nb ) ;
#endif
    RAW_sema = &(RAW_header->raw_sema) ;
    init_data() ;
    *status = RAW_FRESHLY_CREATED ;
  }
  else *status = RAW_NOT_AVAILABLE ;
  return error ;
}

error_code
RAW_reset()
{
  /* A n'appeler qu'a bon escient :
     Si lors d'un RAW_init on a eu dans le status "RAW_ALREADY_EXISTED"
     alors on peut faire un reset.
     Sinon, on peut aussi mais ca ne sert a rien
     */
  error_code ok ;

  /* Tuer le process en attente sur liberation d'un buffer */
  if ( RAW_header->waiting ) _os_send( RAW_header->waiting, 0 ) ;
  if ( RAW_header->spy_waiting ) _os_send( RAW_header->spy_waiting, 0 ) ;
  /* Ce qui suit est peut-etre un peu violent !
     Le process qui fait T1_reset devra auparavant tuer le(s) process en
     attente sur le semaphore. */
  ok = _os_sema_term( RAW_sema ) ;
  /* Maintenant on peut initialiser les donnees */
  init_data() ;
  return ok ;
}

error_code
RAW_deinit()
{
  error_code ok ;
  ok = _os_unlink( RAW_addr ) ;
  if ( ok == RAW_SUCCESS ) {
    RAW_addr = NULL ;
    RAW_header = NULL ;
  }
  return ok ;
}

static error_code
lock_raw()
{
  /* _os_event_wait() */
  return _os_sema_p( RAW_sema ) ;
}

static error_code
unlock_raw()
{
  /* Release semaphore */
  return _os_sema_v( RAW_sema ) ;
}

error_code
RAW_open( RAW_IDENT **bid, void **buff )
{
  int i ;
  RAW_IDENT *pbid ;
  error_code ok ;

  if ( ok = lock_raw() ) return ok ;
  if ( pbid = RAW_header->free_head ) {
    /* RAW_header->free_head contient l'adresse d'un buffer libre
       Marquer ce buffer comme "BUSY".
       decrementer le nb de buffers libres ( nfree-- )
       Changer de liste ( de Free -> Used )
       */
    *buff = pbid->buffer ;
    RAW_header->nfree-- ;
    RAW_header->nused++ ;
    pbid->num = RAW_header->count++ ;
    pbid->use_status = RAW_NOT_READY ;
    pbid->spy_status = RAW_NOT_SPYED ;
    lkl_unlink( pbid,
		(void **)&RAW_header->free_head,
		(void **)&RAW_header->free_tail ) ;
    lkl_head( pbid,
	      (void **)&RAW_header->used_head,
	      (void **)&RAW_header->used_tail ) ;
    *bid = pbid ;
    unlock_raw() ;
    return RAW_SUCCESS ;
  }
  unlock_raw() ;
  return RAW_NO_FREE_BUFFER ;
}

error_code
RAW_open_w( RAW_IDENT **bid, void **buff, signal_code wakeup )
{
  int i ;
  RAW_IDENT *pbid ;
  u_int32 tt ;
  signal_code sig ;
  error_code ok ;

  for( ; ; ) {
    if ( ok = lock_raw() ) return ok ;
    if ( pbid = RAW_header->free_head ) {
      /* T1_header contient l'adresse d'un buffer libre
	 Marquer ce buffer comme "BUSY".
	 decrementer le nb de buffers libres ( nfree-- )
	 Changer de liste ( de Free -> Used )
	 */
      *buff = pbid->buffer ;
      RAW_header->nfree-- ;
      RAW_header->nused++ ;
      pbid->num = RAW_header->count++ ;
      pbid->use_status = RAW_NOT_READY ;
      pbid->spy_status = RAW_NOT_SPYED ;
      lkl_unlink( pbid,
		  (void **)&RAW_header->free_head,
		  (void **)&RAW_header->free_tail ) ;
      lkl_head( pbid,
		(void **)&RAW_header->used_head,
		(void **)&RAW_header->used_tail ) ;
      *bid = pbid ;
      RAW_header->waiting = 0 ;
      unlock_raw() ;
      return RAW_SUCCESS ;
    }
    /* No free buffer, sleep till wakeup signal */
    _os_sigmask( 1 ) ;
    RAW_header->waiting = my_pid() ;
    RAW_header->wakeup = wakeup ;
    tt = 0 ;
    unlock_raw() ;
    do {
      _os_sigmask( 1 ) ;
      _os_sleep( &tt, &sig ) ;
    } while( sig != wakeup ) ;
  }
  /* Should never fall here */
}

error_code
RAW_ready( RAW_IDENT *bid )
{
  error_code ok ;

  if ( ok = lock_raw() ) return ok ;
  if ( bid->use_status == RAW_NOT_READY ) {
    bid->use_status = RAW_NOT_USED ;
    unlock_raw() ;
    /* Reveiller un spy s'il y a lieu */
    if ( RAW_header->spy_waiting )
      _os_send( RAW_header->spy_waiting, RAW_header->spy_wakeup ) ;
    return RAW_SUCCESS ;
  }
  else {
    unlock_raw() ;
    return RAW_NOT_NOT_READY_BUFFER ;
  }
}

error_code
RAW_ready_by_num( int num )
{
  error_code ok ;
  RAW_IDENT *bid ;

  if ( ok = lock_raw() ) return ok ;
  for( bid = RAW_header->used_head ; bid ; bid = bid->next )
    if ( bid->num == num && bid->use_status == RAW_NOT_READY ) {
      bid->use_status = RAW_NOT_USED ;
      unlock_raw() ;
      /* Reveiller un spy s'il y a lieu */
      if ( RAW_header->spy_waiting )
	_os_send( RAW_header->spy_waiting, RAW_header->spy_wakeup ) ;
      return RAW_SUCCESS ;
    }
  unlock_raw() ;
  return RAW_NOT_NOT_READY_BUFFER ;
}

static error_code
delete_buff( RAW_IDENT *pbid )
{
  if ( pbid->use_status == RAW_USED && pbid->spy_status != RAW_IN_SPY ) {
    /* Move the buffer from the Used queue to the Free queue */
    lkl_unlink( pbid,
		(void **)&RAW_header->used_head,
		(void **)&RAW_header->used_tail ) ;
    lkl_tail( pbid,
	      (void **)&RAW_header->free_head,
	      (void **)&RAW_header->free_tail ) ;
    RAW_header->nfree++ ;
    RAW_header->nused-- ;
    /* Wake up waiting process if any */
    if ( RAW_header->waiting )
      _os_send( RAW_header->waiting, RAW_header->wakeup ) ;
  }
  return RAW_SUCCESS ;
}

error_code
RAW_unuse( RAW_IDENT *pbid )
{
  /* delete buffer by its Ident */
  error_code ok ;
  if ( ok = lock_raw() ) return ok ;
  if ( pbid->use_status == RAW_IN_USE ) {
    pbid->use_status = RAW_USED ;
    ok = delete_buff( pbid ) ;
  }
  else ok = RAW_NOT_IN_USE_BUFFER ;
  unlock_raw() ;
  return ok ;
}

error_code
RAW_unuse_by_num( int num )
{
  error_code ok ;
  RAW_IDENT *p1 ;

  if ( ok = lock_raw() ) return ok ;
  for( p1 = RAW_header->used_head ; p1 ; p1 = p1->next )
    if ( p1->num == num && p1->use_status == RAW_IN_USE ) {
      p1->use_status = RAW_USED ;
      ok = delete_buff( p1 ) ;
      unlock_raw() ;
      return ok ;
    }
  unlock_raw() ;
  return RAW_UNKNOWN_BUFFER ;
}

error_code
RAW_unspy( RAW_IDENT *pbid )
{
  /* delete buffer by its Ident */
  error_code ok ;
  if ( ok = lock_raw() ) return ok ;
  if ( pbid->spy_status == RAW_IN_SPY ) {
    pbid->spy_status = RAW_SPYED ;
    ok = delete_buff( pbid ) ;
  }
  else ok = RAW_NOT_IN_SPY_BUFFER ;
  unlock_raw() ;
  return ok ;
}

error_code
RAW_unspy_by_num( int num )
{
  /* Close buffer by its number
       Search the ident of the buffer
       if found, delete
       else error
       */
  error_code ok ;
  RAW_IDENT *p1 ;
  if ( ok = lock_raw() ) return ok ;
  for( p1 = RAW_header->used_head ; p1 ; p1 = p1->next )
    if ( p1->num == num && p1->spy_status == RAW_IN_SPY ) {
      p1->spy_status = RAW_SPYED ;
      ok = delete_buff( p1 ) ;
      unlock_raw() ;
      return ok ;
    }
  unlock_raw() ;
  return RAW_NOT_IN_SPY_BUFFER ;
}

error_code
RAW_spy_old( RAW_IDENT **pb, void **buff )
{
  /* Get the oldest buffer not yet spyed*/
  RAW_IDENT *pbid ;
  error_code ok ;

  if ( ok = lock_raw() ) return ok ;
  for ( pbid = RAW_header->used_tail ; pbid ; pbid = pbid->prev ) {
    if ( pbid->use_status != RAW_NOT_READY &&
	 pbid->spy_status == RAW_NOT_SPYED ) {
      *pb = pbid ;
      *buff = pbid->buffer ;
      pbid->spy_status = RAW_IN_SPY ;
      unlock_raw() ;
      return RAW_SUCCESS ;
    }
  }
  unlock_raw() ;
  /* Should never come here ! */
  return RAW_NO_NO_SPY_BUFFER ;
}

error_code
RAW_spy_old_w( RAW_IDENT **pb, void **buff , signal_code wakeup )
{
  /* Get the oldest buffer not yet spyed*/
  RAW_IDENT *pbid ;
  u_int32 tt ;
  signal_code sig ;
  error_code ok ;

  for( ; ; ) {
    if ( ok = lock_raw() ) return ok ;
    for ( pbid = RAW_header->used_tail ; pbid ; pbid = pbid->prev ) {
      if ( pbid->use_status != RAW_NOT_READY &&
	   pbid->spy_status == RAW_NOT_SPYED ) {
	*pb = pbid ;
	*buff = pbid->buffer ;
	pbid->spy_status = RAW_IN_SPY ;
	RAW_header->spy_waiting = 0 ;
	unlock_raw() ;
	return RAW_SUCCESS ;
      }
    }
    unlock_raw() ;
    /* No free buffer, sleep till wakeup signal */
    _os_sigmask( 1 ) ;
    RAW_header->spy_waiting = my_pid() ;
    RAW_header->spy_wakeup = wakeup ;
    tt = 0 ;
    unlock_raw() ;
    do {
      _os_sigmask( 1 ) ;
      _os_sleep( &tt, &sig ) ;
    } while( sig != wakeup ) ;
  }
  /* Should never come here ! */
  return RAW_NO_NO_SPY_BUFFER ;
}

error_code
RAW_use( RAW_IDENT **pb, void **buff )
{
  /* Get the oldest not yet used buffer */
  RAW_IDENT *pbid ;
  error_code ok ;

  if ( ok = lock_raw() ) return ok ;
  for ( pbid = RAW_header->used_tail ; pbid ; pbid = pbid->prev )
  if ( pbid && pbid->use_status == RAW_NOT_USED ) {
    *pb = pbid ;
    *buff = pbid->buffer ;
    pbid->use_status = RAW_IN_USE ;
    unlock_raw() ;
    return RAW_SUCCESS ;
  }
  unlock_raw() ;
  /* Should never come here ! */
  return RAW_NO_OPENED_BUFFER ;
}

void
RAW_buff_dump( FILE *fout, int part )
{
  RAW_IDENT *p1 ;

  fprintf( fout, "Number of buffers : %d\n", RAW_header->nbuffs ) ;
  fprintf( fout, "Number of free    : %d\n", RAW_header->nfree ) ;
  fprintf( fout, "Number of used    : %d\n", RAW_header->nused ) ;
  if ( part == 0 ) {
    fprintf( fout, "Free buffers chain\n" ) ;
    for( p1 = RAW_header->free_head ; p1 ; p1 = p1->next )
      fprintf( fout, "   Address : %x\n", p1->buffer ) ;
    fprintf( fout, "Used buffers chain ( newest first )\n" ) ;
    for( p1 = RAW_header->used_head ; p1 ; p1 = p1->next )
      fprintf( fout, "   Address : %x - Ident : %d - Status :%s + %s\n",
	       p1->buffer, p1->num,
	       Str_use_status[p1->use_status],
	       Str_spy_status[p1->spy_status] ) ;
  }
}

