#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>
#include <sys/msg.h>
#include <signal.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <errno.h>

#if defined(__GNU_LIBRARY__) && !defined(_SEM_SEMUN_UNDEFINED)
/* union semun is defined by including <sys/sem.h> */
#else
/* according to X/OPEN we have to define it ourselves */
union semun {
    int val;                    /* value for SETVAL */
    struct semid_ds *buf;       /* buffer for IPC_STAT, IPC_SET */
    unsigned short int *array;  /* array for GETALL, SETALL */
    struct seminfo *__buf;      /* buffer for IPC_INFO */
};
#endif

struct  mymsgbuf  {
          long mtype;
          int wincr;
	  int sincr;
	  int linkcount;
};


/************** os9 needed typedef's ********************/
typedef void mh_data;
typedef void mh_com;
typedef int error_code;
typedef int path_id;
typedef int semaphore;
typedef int int32;
typedef unsigned int event_id;
typedef unsigned int alarm_id;
typedef unsigned int signal_code;
typedef unsigned int u_int32;
typedef unsigned short u_int16;
typedef pid_t process_id;


void *_glob_data=0;
u_int32 *_os_fork=0;


/********************************************************/
int os9_lin_perm( int perm )
{
/*this function transforms os9 perms format into linux  */
/*only takes the rwe permisions                         */

int owner_r, owner_w, owner_e;
int group_r, group_w, group_e;
int other_r, other_w, other_e;

owner_r = (perm & 0x0001) << 8;
owner_w = (perm & 0x0002) << 6;
owner_e = (perm & 0x0004) << 4;
group_r = (perm & 0x0010) << 1;
group_w = (perm & 0x0020) >> 1;
group_e = (perm & 0x0040) >> 3;
other_r = (perm & 0x0100) >> 6;
other_w = (perm & 0x0200) >> 8;
other_e = (perm & 0x0400) >> 10;
return (owner_r|owner_w|owner_e|group_r|group_w|group_e|other_r|other_w|other_e);
}





/********************************************************/
error_code _os_alarm_cycle( alarm_id *alrm_id, 
		            signal_code signal, 
			    u_int32 time )
{
/*this function uses the linux setitimer function       */
/*and it does not take care of the alarm_id and the     */
/*signal_code, only use the default signal SIGALRM      */

struct itimerval temp;

if( time & 0x80000000 ) {
    time = (time - 0x80000000) / 256;
    temp.it_interval.tv_sec = time; 
    temp.it_interval.tv_usec = 0;
    temp.it_value.tv_sec = time;
    temp.it_value.tv_usec = 0;
}
else {
    temp.it_interval.tv_sec = 0; 
    temp.it_interval.tv_usec = time;
    temp.it_value.tv_sec = 0;
    temp.it_value.tv_usec = time;
}        
setitimer(ITIMER_REAL, &temp, NULL);
return 0;
}



/********************************************************/
error_code _os_alarm_set( alarm_id *alrm_id, 
		          signal_code signal, 
			  u_int32 time )
{
/*this function uses the linux alarm function           */
/*and it does not take care of the alarm_id and the     */
/*signal_code, only use the default signal SIGALRM      */

if( time & 0x80000000 ) time = (time - 0x80000000) / 256;
alarm(time);
return 0;
}



/********************************************************/
error_code _os_alarm_delete( alarm_id *alrm_id ) 
{
/*this function uses the linux alarm function           */
/*and the linux setitimer function to stop the timer    */
/*and it does not take care of the alarm_id             */

struct itimerval temp;

temp.it_interval.tv_sec = 0; 
temp.it_interval.tv_usec = 0;
temp.it_value.tv_sec = 0;
temp.it_value.tv_usec = 0;

alarm(0);
setitimer(ITIMER_REAL, &temp, NULL);
return 0;
}




/********************************************************/
error_code _os_close( path_id path ) 
{
/*this function uses the linux close function           */

return ( close(path) );
}





/********************************************************/
error_code _os_datmod( char *mod_name,
		       u_int32 size,
		       u_int16 *attr_rev,
		       u_int16 *type_lang,     
		       u_int32 perm,
		       void **mod_exec,
		       mh_data **mod_head )
{
/*this function uses the SV IPC functions               */
/*to simulate data modules with shared memory segments  */
/*Be aware that it does not take care of attr_rev, type */
/*lang and perm. mod_exec and mod_head points the same  */

key_t key;           
int shmid;
char newstr[6];
char *shmptr;
                            /*take the first five chars */
strncpy(newstr,mod_name,5); /*to make the key           */
newstr[5]='\0';
key = (key_t)strtol(newstr, (char **)NULL, 36);
if( (shmid = shmget(key, size, IPC_CREAT|IPC_EXCL|0666)) ==  -1 ) {
    if( (shmid = shmget(key, size, 0666)) == -1 ) {
	return -1;
    }
}
if( (shmptr = shmat(shmid, 0, 0)) == (char *)-1 ) return -1;
*mod_exec = shmptr;
*mod_head = shmptr;
return 0;
}





/********************************************************/
error_code _os_link( char **mod_name,
		     mh_com **mod_head,
		     void **mod_exec,
		     u_int16 *type_lang,
		     u_int16 *attr_rev )
{
/*this function uses the SV IPC functions               */
/*to link to shared memory segments                     */
/*Be aware that it does not take care of attr_rev, type */
/*lang.                                                 */

key_t key;           
int shmid;
char newstr[6];
char *shmptr;
                            /*take the first five chars */
strncpy(newstr,*mod_name,5); /*to make the key          */
newstr[5]='\0';
key = (key_t)strtol(newstr, (char **)NULL, 36);
if( (shmid = shmget(key, 0, 0666)) == -1 ) {
    return -1;
    }
if( (shmptr = shmat(shmid, 0, 0)) == (char *)-1 ) return -1;
*mod_exec = shmptr;
*mod_head = shmptr;
return 0;
}





/********************************************************/
error_code _os_unlink( mh_com *mod_head )
{
/*this function uses the SV IPC functions               */
/*to unlink shared memory segments                      */

return shmdt(mod_head);
}





/********************************************************/
error_code _os_dup( path_id dup_path,
		    path_id *new_path ) 
{
/*this function uses the linux dup function             */

int p;

if ( (p = dup(dup_path)) >= 0 ) {
    *new_path = p;
    return 0;
    }
    else return p;
}





/********************************************************/
error_code _os_exec( u_int32 *func,
		     u_int32 priority,
		     u_int32 pathcnt,
		     void *modname,
		     char **argv,
		     char **envp,
		     u_int32 datasize,
		     process_id *proc_id ) 
{
/*this function uses the linux execvp function          */
/*and it's oriented to be used with the fork function   */
/*so it does not take care of the passed function       */

pid_t childpid;

if ( (childpid = fork()) == 0 ) {/*****I'm the child*****/
    execvp( (char *)modname, argv );
    }
    else
	{
	if ( childpid < 0 ) {    /*********error*********/
	    return childpid;
	}
	else {
	    *proc_id = childpid;
	    }
	}
return 0;
}    




/********************************************************/
error_code _os_read( path_id path,
		     void *buffer,
		     u_int32 *count )
{
/*this function uses the linux read function            */		     

int r;

if ( (r = read(path, buffer, *count)) >= 0 ) {
    *count = r;
    return 0;
    }
    else return r;
}    
    




/********************************************************/
error_code _os_write( path_id path,
		     void *buffer,
		     u_int32 *count )
{
/*this function uses the linux write function           */		     

int w;

if ( (w = write(path, buffer, *count)) >= 0 ) {
    *count = w;
    return 0;
    }
    else return w;
}




/********************************************************/
error_code _os_rte( void )
{
/*this function does nothing under linux                */		     

return 0;
}





/********************************************************/
error_code _os_send( process_id proc_id,
		     signal_code signal )
{
/*this function uses the kill linux function            */		     

return ( kill(proc_id, signal) );
}





/********************************************************/
error_code _os_sema_init( semaphore *sema )
{
/*this function uses the SV IPC functions               */
/*to create semaphores. Be aware that no 0s9 full       */
/*compatibility is implemented, only P and V of a binary*/
/*semaphore                                             */ 

key_t key;           
union semun semopts;
int sid;

key = (key_t)sema;

if((sid = semget(key, 1, IPC_CREAT|IPC_EXCL|0666)) ==  -1) {
    if((sid = semget(key, 0, 0666)) == -1) {
	return -1;
    }
}
semopts.val = 1;
semctl(sid, 0, SETVAL, semopts);
return 0;
}





/********************************************************/
error_code _os_sema_p( semaphore *sema )
{
/*this function uses the SV IPC functions               */
/*to lock semaphores. Be aware that no 0s9 full         */
/*compatibility is implemented, only P and V of a binary*/
/*semaphore                                             */ 

key_t key;           
struct sembuf sem_lock = { 0, -1, SEM_UNDO};
int sid;

key = (key_t)sema;

if((sid = semget(key, 0, 0666)) == -1) {
    return -1;
}

if( (semop(sid, &sem_lock, 1)) == -1 ) return -1;
return 0; 
}




/********************************************************/
error_code _os_sema_v( semaphore *sema )
{
/*this function uses the SV IPC functions               */
/*to unlock semaphores. Be aware that no 0s9 full       */
/*compatibility is implemented, only P and V of a binary*/
/*semaphore                                             */ 

key_t key;           
struct sembuf sem_unlock = { 0, 1, SEM_UNDO};
int sid, semval;

key = (key_t)sema;

if((sid = semget(key, 0, 0666)) == -1) {
    return -1;
}

/*Esta bloqueado?*/
if( (semval = semctl(sid, 0, GETVAL, 0)) == 1 ) return -1;

/*Intentamos desbloquear */
if((semop(sid, &sem_unlock, 1)) == -1) return -1;
return 0;
}





/********************************************************/
error_code _os_sema_term( semaphore *sema )
{
/*this function uses the SV IPC functions               */
/*to destroy semaphores.                                */

key_t key;           
int sid;

key = (key_t)sema;

if((sid = semget(key, 0, 0666)) == -1) {
    return -1;
}
semctl(sid, 0, IPC_RMID, 0);
return 0;
}






/********************************************************/
error_code _os_getime( time_t *tim,
		       u_int32 *ticks )
{
/*this function uses the time linux function            */
/*Be aware that the tick functonality is not implemented*/

time_t t;

if( (t = time(tim)) == (time_t)-1 ) return t;
else return 0;
}




/********************************************************/
error_code _os_ev_creat( int32 winc,
		         int32 sinc,
		         u_int32 perm,
		         event_id *ev_id,
		         char *name,
			 int32 value,
			 u_int32 color )
{
/*this function uses the SV IPC functions               */
/*to simulate events with messages queues.              */
/*0s9 full compatibility is implemented.                */

key_t key;           
char newstr[6];
int msg_id;
struct  mymsgbuf buf ;
struct  mymsgbuf *qbuf ;
qbuf=&buf;
                           /*take the first five chars */
strncpy(newstr,name,5);    /*to make the key           */
newstr[5]='\0';
key = (key_t)strtol(newstr, (char **)NULL, 36);

if((msg_id  =  msgget(key,  IPC_CREAT|IPC_EXCL|0666))  ==  -1)  {
    if((msg_id = msgget(key, 0666))  ==  -1) {    
	return -1;
    }
}
else {
    qbuf->mtype = value + 0x1000000;
    qbuf->wincr = winc;
    qbuf->sincr = sinc;
    qbuf->linkcount = 0;
    msgsnd(msg_id, (struct msgbuf *)qbuf, 2*sizeof(int), 0);
    }
*ev_id = msg_id;    
return 0;
}





/********************************************************/
error_code _os_ev_delete( char *name )
{
/*this function uses the SV IPC functions               */
/*to destroy events (messages queues).                  */

key_t key;           
int qid;
char newstr[6];
struct  mymsgbuf buf ;
struct  mymsgbuf *qbuf ;
qbuf=&buf;

                           /*take the first five chars */
strncpy(newstr,name,5);    /*to make the key           */
newstr[5]='\0';
key = (key_t)strtol(newstr, (char **)NULL, 36);

if((qid = msgget(key, 0666)) == -1) {
    return -1;
}

msgrcv(qid, (struct msgbuf *)qbuf, 2*sizeof(int), 0, 0);
if(qbuf->linkcount == 0) {
    msgctl(qid, IPC_RMID, 0);
    }
    else {
	msgsnd(qid, (struct msgbuf *)qbuf, 2*sizeof(int), 0);
	}

return 0;
}





/********************************************************/
error_code _os_ev_link( char *name,
			event_id *ev_id )
{
/*this function uses the SV IPC functions               */
/*to link to an event (message queue).                  */

key_t key;           
int qid;
char newstr[6];
struct  mymsgbuf buf ;
struct  mymsgbuf *qbuf ;
qbuf=&buf;
                           /*take the first five chars */
strncpy(newstr,name,5);    /*to make the key           */
newstr[5]='\0';
key = (key_t)strtol(newstr, (char **)NULL, 36);

if((qid = msgget(key, 0666)) == -1) {
    return -1;
}
msgrcv(qid, (struct msgbuf *)qbuf, 2*sizeof(int), 0, 0);
qbuf->linkcount += 1;
msgsnd(qid, (struct msgbuf *)qbuf, 2*sizeof(int), 0);
*ev_id = qid;
return 0;
}





/********************************************************/
error_code _os_ev_read( event_id ev_id,
			int32 *value )
{
/*this function uses the SV IPC functions               */
/*to get the value of an event (message queue).         */

struct  mymsgbuf buf ;
struct  mymsgbuf *qbuf ;
qbuf=&buf;

msgrcv(ev_id, (struct msgbuf *)qbuf, 2*sizeof(int), 0, 0);
msgsnd(ev_id, (struct msgbuf *)qbuf, 2*sizeof(int), 0);

*value = qbuf->mtype - 0x1000000;
return 0;
}





/********************************************************/
error_code _os_ev_signal( event_id ev_id,
			  int32 *value,
			  u_int32 actv_flag )
{
/*this function uses the SV IPC functions               */
/*to signal events.                                     */

struct  mymsgbuf buf ;
struct  mymsgbuf *qbuf ;
qbuf=&buf;

msgrcv(ev_id, (struct msgbuf *)qbuf, 2*sizeof(int), 0, 0);

*value = qbuf->mtype - 0x1000000;    
qbuf->mtype += qbuf->sincr; 
msgsnd(ev_id, (struct msgbuf *)qbuf, 2*sizeof(int), 0);

return 0;
}






/********************************************************/
error_code _os_ev_unlink( event_id ev_id )
{
/*this function uses the SV IPC functions               */
/*to unlink to an event.                                */

struct  mymsgbuf buf ;
struct  mymsgbuf *qbuf ;
qbuf=&buf;

msgrcv(ev_id, (struct msgbuf *)qbuf, 2*sizeof(int), 0, 0);
qbuf->linkcount -= 1;
msgsnd(ev_id, (struct msgbuf *)qbuf, 2*sizeof(int), 0);
return (qbuf->linkcount);
}






/********************************************************/
error_code _os_ev_wait( event_id ev_id,
			int32 *value,
			signal_code *signal,
			int32 min_val,
			int32 max_val )
{
/*this function uses the SV IPC functions               */
/*to wait events.                                       */

struct timespec tmp1;
int found = 0;
int res;
int i;
struct  mymsgbuf buf ;
struct  mymsgbuf *qbuf ;
qbuf=&buf;
tmp1.tv_sec = 0; 
tmp1.tv_nsec = 20000000;

while(!found)
    {
    for(i=min_val;i<=max_val;i++)    
	{
	    if ((res = msgrcv(ev_id, NULL, 0, i, IPC_NOWAIT)) == -1)
		{
		    if (errno == E2BIG) 
			{
			found = 1;
			msgrcv(ev_id, (struct msgbuf *)qbuf, 2*sizeof(int), i, 0);
			qbuf->mtype += qbuf->wincr; 
			if ((msgsnd(ev_id, (struct msgbuf *)qbuf, 2*sizeof(int), 0)) == -1)
			    {
			    return -1;
			    }
			break;
			}
		}
	}
    nanosleep(&tmp1,NULL);
    }
*value = qbuf->mtype - 0x1000000;
return 0; 
}






/********************************************************/
error_code _os_ev_allset( event_id ev_id,
			  int32 *value,
			  signal_code *signal,
			  u_int32 mask )
{
/*this function uses the SV IPC functions               */

struct timespec tmp2;
struct  mymsgbuf buf ;
struct  mymsgbuf *qbuf ;
qbuf = &buf;
mask = mask + 0x1000000;
tmp2.tv_sec = 0; 
tmp2.tv_nsec = 20000000;

while( (qbuf->mtype & mask) != mask) {
    msgrcv(ev_id, (struct msgbuf *)qbuf, 2*sizeof(int), 0, 0);
    msgsnd(ev_id, (struct msgbuf *)qbuf, 2*sizeof(int), 0);
    nanosleep(&tmp2,NULL);
    }
*value = qbuf->mtype - 0x1000000;
return 0;
}






/********************************************************/
error_code _os_ev_setand( event_id ev_id,
			  int32 *value,
			  u_int32 mask,
			  u_int32 actv_flag )
{
/*this function uses the SV IPC functions               */

struct  mymsgbuf buf ;
struct  mymsgbuf *qbuf ;
qbuf=&buf;

msgrcv(ev_id, (struct msgbuf *)qbuf, 2*sizeof(int), 0, 0);
*value = qbuf->mtype - 0x1000000;
qbuf->mtype = qbuf->mtype & (long)mask;    
msgsnd(ev_id, (struct msgbuf *)qbuf, 2*sizeof(int), 0);
return 0;
}






/********************************************************/
error_code _os_ev_setor ( event_id ev_id,
			  int32 *value,
			  u_int32 mask,
			  u_int32 actv_flag )
{
/*this function uses the SV IPC functions               */

struct  mymsgbuf buf ;
struct  mymsgbuf *qbuf ;
qbuf=&buf;

msgrcv(ev_id, (struct msgbuf *)qbuf, 2*sizeof(int), 0, 0);
*value = qbuf->mtype - 0x1000000;
qbuf->mtype = qbuf->mtype | (long)mask;    
msgsnd(ev_id, (struct msgbuf *)qbuf, 2*sizeof(int), 0);
return 0;
}






/********************************************************/
error_code _os_sleep( u_int32 *ticks,
		      signal_code *signl )
{
/*this function uses the sleep and nanosleep functions  */
/*ticks are simulated by number of time slices (10ms)   */
struct timespec tmp;

if( (*ticks) == 0 ) sleep(10000);
else if( (*ticks) & 0x80000000 ) {
    *ticks = ((*ticks) - 0x80000000) / 256;
    sleep(*ticks);
}
else {
    tmp.tv_sec = 0; 
    tmp.tv_nsec = (*ticks) * 10000000;
    nanosleep(&tmp,NULL);
}
return 0;
}





/********************************************************/
int intercept( void (*icpthand)(int) )
{
/*this function intercepts only the alarm signal, by now*/
int i;
signal(3, (*icpthand));
signal(14, (*icpthand));
for(i=33; i<48; i++) {
    signal(i, (*icpthand));
    }
//signal(SIGALRM, SIG_DFL);
return 0;
}






/********************************************************/
error_code _os_intercept( void (*icpthand)(int),
			  void *data_ptr )
{
/*this function intercepts only the alarm signal, by now*/
int i;
signal(3, (*icpthand));
signal(14, (*icpthand));
for(i=33; i<48; i++) {
    signal(i, (*icpthand));
    }
//signal(SIGALRM, SIG_DFL);
return 0;
}






/********************************************************/
int _errmsg( int nerr,
	     char *msg,
	     char *name )
{
/*this function uses the perror linux function          */
perror(msg);
return nerr;
}






/********************************************************/
error_code _os_id( process_id *proc_id,
		   u_int16 *priority,
		   u_int16 *age,
		   int32 *schedule,
		   u_int16 *group,
		   u_int16 *user )
{
/*this function uses the getpid linux function          */
*proc_id = getpid();
return 0;
}





/********************************************************/
error_code _os_sigmask( int32 mode )
{
/*this function uses the sigsetmask linux function      */
static int mask;

if ( mode == 0 ) {
    sigsetmask(0);
    mask=0;
}
else {
    mask=mask+mode;
    if ( mask <= 0 ) {
	mask=0;     
	sigsetmask(0);
    }
    else sigsetmask(0xffff);
}    	
return 0;
}










